const { Op, Sequelize } = require("sequelize");
const {
  CrmVis_record,
  CrmConv_record,
  CrmFu_record,
  CrmApt_record,
} = require("../models/Kyc");
const { PR_patientReg } = require("../models/concentModel.js");
const { KYC } = require("../models/Kyc");
const {
  treatmentHist,
  PatientCounseling,
  EmbryoTransfer,
  treatmentAdvice,
  BetaHcg,
} = require("../models/embrology");
const { ClinicConfiguration } = require("../models/clinicConfig");
const { sequelize } = require("../sequelize.js");

const getRecord = async (startDate, endDate, req) => {
  // Step 1: Fetch all clinics
  const clinicRecords = await ClinicConfiguration.findAll({
    where: {
      clinic_id: req.user.clinicId,
    },
    attributes: ["id", "clinic_id", "clinic_desc", "clinic_area"],
    raw: true,
  });

  // Step 2: Fetch all KYC records (no date filtering)
  const kycRecords = await KYC.findAll({
    attributes: ["id", "clinic_id"],
    raw: true,
  });

  const kycIds = kycRecords.map((kyc) => kyc.id);

  // Step 3: Fetch CRM data for the given KYC IDs with date filtering
  const crmData = {
    visits: await CrmVis_record.findAll({
      where: {
        kyc_id: { [Op.in]: kycIds },
        createdAt: {
          [Op.between]: [startDate, endDate],
        },
      },
      attributes: [
        "kyc_id",
        [Sequelize.fn("COUNT", Sequelize.col("kyc_id")), "total_visits"],
      ],
      group: ["kyc_id"],
      raw: true,
    }),
    conversions: await CrmConv_record.findAll({
      where: {
        kyc_id: { [Op.in]: kycIds },
        createdAt: {
          [Op.between]: [startDate, endDate],
        },
      },
      attributes: [
        "kyc_id",
        [Sequelize.fn("COUNT", Sequelize.col("kyc_id")), "total_conversions"],
      ],
      group: ["kyc_id"],
      raw: true,
    }),
    followups: await CrmFu_record.findAll({
      where: {
        kyc_id: { [Op.in]: kycIds },
        createdAt: {
          [Op.between]: [startDate, endDate],
        },
      },
      attributes: [
        "kyc_id",
        [Sequelize.fn("COUNT", Sequelize.col("kyc_id")), "total_followups"],
      ],
      group: ["kyc_id"],
      raw: true,
    }),
    appointments: await CrmApt_record.findAll({
      where: {
        kyc_id: { [Op.in]: kycIds },
        createdAt: {
          [Op.between]: [startDate, endDate],
        },
      },
      attributes: [
        "kyc_id",
        [Sequelize.fn("COUNT", Sequelize.col("kyc_id")), "total_appointments"],
      ],
      group: ["kyc_id"],
      raw: true,
    }),
  };

  // Get month name for CRM record date range
  const monthName = startDate.toLocaleString("default", { month: "long" });

  // Step 4: Combine CRM data with clinic IDs and ensure default zero values
  const clinicData = clinicRecords.map((clinic) => {
    const kycIdsForClinic = kycRecords
      .filter((kyc) => kyc.clinic_id === clinic.id.toString())
      .map((kyc) => kyc.id);

    const visits = crmData.visits
      .filter((record) => kycIdsForClinic.includes(parseInt(record.kyc_id)))
      .reduce((sum, record) => sum + parseInt(record.total_visits), 0);

    const conversions = crmData.conversions
      .filter((record) => kycIdsForClinic.includes(parseInt(record.kyc_id)))
      .reduce((sum, record) => sum + parseInt(record.total_conversions), 0);

    const followups = crmData.followups
      .filter((record) => kycIdsForClinic.includes(parseInt(record.kyc_id)))
      .reduce((sum, record) => sum + parseInt(record.total_followups), 0);

    const appointments = crmData.appointments
      .filter((record) => kycIdsForClinic.includes(parseInt(record.kyc_id)))
      .reduce((sum, record) => sum + parseInt(record.total_appointments), 0);

    return {
      clinic_id: clinic.id,
      clinic_name: clinic.clinic_desc,
      clinic_area: clinic.clinic_area,
      total_kyc: kycIdsForClinic.length || 0, // No date filtering applied
      total_visits: visits || 0, // Filtered by date
      total_conversions: conversions || 0, // Filtered by date
      total_followups: followups || 0, // Filtered by date
      total_appointments: appointments || 0, // Filtered by date
      month: monthName, // Derived from the CRM date range
    };
  });

  return clinicData;
};

const getCrmRecords = async (req, res) => {
  try {
    const { from, to, weekly, monthly } = req.query; // Get query params

    const clinics = await ClinicConfiguration.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["id", "clinic_id"],
      raw: true,
    });

    let clinicIds = clinics.map(({ id }) => id);

    // Build conditions dynamically
    const dateCondition =
      from && to
        ? {
            createdAt: {
              [Op.between]: [new Date(from), new Date(to)],
              clinic_id: {
                [Op.in]: clinicIds,
              },
            },
          }
        : {
            clinic_id: {
              [Op.in]: clinicIds,
            },
          };

    // Use the condition in the counts
    const rowCount = await KYC.count({ where: dateCondition });
    const visitsCount = await CrmVis_record.count({ where: dateCondition });
    const convsCount = await CrmConv_record.count({ where: dateCondition });
    const repatsCount = await CrmApt_record.count({ where: dateCondition });
    if ((!from || !to) && !weekly && !monthly) {
      const today = new Date();
      const startOfToday = new Date(today);
      startOfToday.setHours(0, 0, 0, 0); // Start of day
      const endOfToday = new Date(today);
      endOfToday.setHours(23, 59, 59, 999); // End of day
      let result = [await getRecord(startOfToday, endOfToday, req)]; // Fetch data for today\
      res.status(200).json({
        result,
        leadCount: rowCount,
        visitsCount,
        convsCount,
        repatsCount,
      });
      return;
    }

    // Initialize variables for results
    let records = [];

    if (from && to) {
      const startDate = new Date(from);
      const endDate = new Date(to);
      endDate.setHours(23, 59, 59, 999); // Set end of the day for the `to` date

      records = [await getRecord(startDate, endDate, req)];
      res.status(200).json({
        records,
        leadCount: rowCount,
        visitsCount,
        convsCount,
        repatsCount,
      });
      return;
    }

    if (weekly) {
      let record = {};
      const weeksAgo = parseInt(weekly, 10);
      if (!isNaN(weeksAgo) && weeksAgo > 0) {
        for (let i = 0; i < weeksAgo; i++) {
          const today = new Date();
          const endOfWeek = new Date(today);
          endOfWeek.setDate(today.getDate() - 7 * i); // Adjust for i weeks ago
          endOfWeek.setHours(23, 59, 59, 999); // End of day

          const startOfWeek = new Date(endOfWeek);
          startOfWeek.setDate(endOfWeek.getDate() - 6); // Start of the week (7 days ago)
          startOfWeek.setHours(0, 0, 0, 0); // Start of day

          // Build where condition for the current week's date range

          record = await getRecord(startOfWeek, endOfWeek, req);
          records.push(record);
        }
      }
      res.status(200).json({
        records,
        leadCount: rowCount,
        visitsCount,
        convsCount,
        repatsCount,
      });
      return;
    }

    if (monthly) {
      let record = {};
      const monthsAgo = parseInt(monthly, 10);
      if (!isNaN(monthsAgo) && monthsAgo > 0) {
        for (let i = 0; i < monthsAgo; i++) {
          const today = new Date();

          // Calculate endOfMonth and startOfMonth dates
          const endOfMonth = new Date(today);
          endOfMonth.setMonth(today.getMonth() - i); // Adjust for i months ago
          endOfMonth.setDate(
            new Date(
              endOfMonth.getFullYear(),
              endOfMonth.getMonth() + 1,
              0
            ).getDate()
          ); // End of the month
          endOfMonth.setHours(23, 59, 59, 999); // End of day

          const startOfMonth = new Date(endOfMonth);
          startOfMonth.setDate(1); // Start of the month
          startOfMonth.setHours(0, 0, 0, 0); // Start of day

          // Build where condition for the current month's date range

          record = await getRecord(startOfMonth, endOfMonth, req);
          records.push(record);
        }
      }

      console.log(`Total rows in KYC table: ${rowCount}`);
      res.status(200).json({
        records,
        leadCount: rowCount,
        visitsCount,
        convsCount,
        repatsCount,
      });
      return;
    }
  } catch (err) {
    console.error("Error fetching data:", err);
    res.status(500).send("Server Error");
  }
};

const getDashboardData = async (req, res) => {
  try {
    const clinicId = req.query.id; // Retrieve clinicId from req.query.id
    const { from, to } = req.query; // Retrieve 'from' and 'to' date range from query

    // Define months for grouping
    const months = [
      "January",
      "February",
      "March",
      "April",
      "May",
      "June",
      "July",
      "August",
      "September",
      "October",
      "November",
      "December",
    ];

    // Function to fetch data for a specific clinic
    const fetchClinicData = async (id) => {
      // Fetch total records for PatientCounseling
      const totalCounseling = await PatientCounseling.count({
        where: { clinicId: id },
      });

      const treatmentTypes = [
        "Ovum-Freezing",
        "Sperm-Freezing",
        "Embryo-Freezing",
      ];

      const treatments = await treatmentAdvice.findAll({
        where: { clinic_id: id },
        attributes: ["clinic_id", "treatment", "patientId"],
        raw: true,
      });

      let footFallData = { infertile: 0, others: 0 };

      const withDesiredTypes = treatments.filter((record) =>
        record.treatment.some((treatment) =>
          treatmentTypes.includes(treatment.treatmentType)
        )
      );

      const withoutDesiredTypes = treatments.filter(
        (record) =>
          !record.treatment.some((treatment) =>
            treatmentTypes.includes(treatment.treatmentType)
          )
      );

      footFallData.infertile = withoutDesiredTypes.length;
      footFallData.others = withDesiredTypes.length;

      // Fetch total records for PR_patientReg
      const totalPatients = await PR_patientReg.count({
        where: { clinic_id: id },
      });

      // Fetch all records for patient registration
      let records = await PR_patientReg.findAll({
        where: { clinic_id: id },
      });

      // If 'from' and 'to' date range are provided, filter the records
      if (from && to) {
        const fromDate = new Date(from);
        const toDate = new Date(to);
        records = records.filter((record) => {
          const createdAt = new Date(record.createdAt);
          return createdAt >= fromDate && createdAt <= toDate;
        });
      }

      // Get current date and time
      const now = new Date();
      const todayStart = new Date(
        now.getFullYear(),
        now.getMonth(),
        now.getDate()
      );
      const yesterdayStart = new Date(todayStart);
      yesterdayStart.setDate(yesterdayStart.getDate() - 1);

      // Filter records for today
      const todayCount = records.filter((record) => {
        const createdAt = new Date(record.createdAt);
        return createdAt >= todayStart;
      }).length;

      // Filter records for yesterday
      const yesterdayCount = records.filter((record) => {
        const createdAt = new Date(record.createdAt);
        return createdAt >= yesterdayStart && createdAt < todayStart;
      }).length;

      let dod =
        yesterdayCount > 0
          ? ((todayCount - yesterdayCount) / yesterdayCount) * 100
          : todayCount > 0
          ? 100
          : 0;

      // Group the records by month
      const monthCount = months.reduce(
        (acc, month) => ({ ...acc, [month]: 0 }),
        {}
      );
      records.forEach((record) => {
        const createdAt = new Date(record.createdAt);
        const monthName = months[createdAt.getMonth()];
        monthCount[monthName] += 1;
      });

      // Fetch all package_id values for patient counseling
      const patient_counseling = await PatientCounseling.findAll({
        where: { clinicId: id },
        raw: true,
      });

      // Transform the package counts into the required format
      const packageSummary = {
        IUI: 0,
        IVF: 0,
        ICSI: 0,
        "Embro-Freezing": 0,
        "Sperm-Freezing": 0,
        "Ovum-Freezing": 0,
        OPU: 0,
      };
      let convertedPackageSummary = {
        IUI: 0,
        IVF: 0,
        ICSI: 0,
        "Embro-Freezing": 0,
        "Sperm-Freezing": 0,
        "Ovum-Freezing": 0,
        OPU: 0,
      };
      treatments.forEach((t) => {
        t.treatment.forEach((t) => {
          packageSummary[t.treatmentType] += 1;
        });
      });

      patient_counseling.forEach(
        (p) => (convertedPackageSummary[p.treatmentType] += 1)
      );

      let patientDataIvf = {};
      treatments.forEach((p) => {
        p.treatment.forEach((t) => {
          if (t.treatmentType === "IVF") {
            if (patientDataIvf[p.patientId] !== undefined) {
              patientDataIvf[p.patientId]++;
            } else patientDataIvf[p.patientId] = 1;
          }
        });
      });

      let ivfPackageData = {
        oneTime: 0,
        twoTime: 0,
        threeTime: 0,
        fourTime: 0,
        fiveTime: 0,
      };
      for (const [key, value] of Object.entries(patientDataIvf)) {
        if (value === 1) ivfPackageData.oneTime++;
        else if (value === 2) ivfPackageData.twoTime++;
        else if (value === 3) ivfPackageData.threeTime++;
        else if (value === 4) ivfPackageData.fourTime++;
        else ivfPackageData.fiveTime++;
      }

      let pId = records.map((r) => r.id);
      const embryoTransfer = await EmbryoTransfer.count({
        where: {
          patientId: {
            [Op.in]: pId,
          },
        },
      });

      const fileToEt = {
        files: records.length,
        et: embryoTransfer,
      };
      const Hcg = await BetaHcg.findAll({
        attributes: ["betahcgValue"],
        where: { clinicId: id },
      });
      let betHcgValue = {
        positive: 0,
        negative: 0,
      };
      Hcg.forEach((hcg) => {
        if (hcg.betahcgValue >= 5) betHcgValue.positive++;
        else betHcgValue.negative++;
      });

      let fileData = {
        consult: 0,
        register: 0,
        file: 0,
      };

      fileData.consult = patient_counseling.length;
      fileData.register = records.length;
      let fileNum = await PatientCounseling.count({
        where: {
          patientId: {
            [Op.in]: pId,
          },
        },
      });
      fileData.file = fileNum;

      return {
        clinicId: id,
        totalPatients,
        TotalCounselingDone: totalCounseling,
        monthlyPatientCount: monthCount,
        packageSummary,
        convertedPackageSummary,
        footFallData,
        dod,
        ivfPackageData,
        fileToEt,
        betHcgValue,
        fileData,
      };
    };

    let responseData;
    console.log("clinic id: ", clinicId);
    if (clinicId) {
      console.log("clinic id: ", clinicId);
      // Fetch data for a single clinic
      responseData = [await fetchClinicData(clinicId)];
    } else {
      // Fetch all unique clinic IDs
      const clinicIds = await ClinicConfiguration.findAll({
        where: {
          clinic_id: req.user.clinicId,
        },
        attributes: ["clinic_id", "id"],
        raw: true,
      });

      // Fetch data for all clinics and return as an array
      responseData = await Promise.all(
        clinicIds.map(({ id }) => fetchClinicData(id))
      );
    }

    res.status(200).json({
      success: true,
      data: responseData,
    });
  } catch (error) {
    console.error("Error fetching dashboard data:", error);
    res.status(500).json({
      success: false,
      message: "An error occurred while fetching the dashboard data.",
    });
  }
};




module.exports = { getCrmRecords, getDashboardData };
