const { OpeningBalance, OpeningBalanceItem, Indent, IndentItem, Requisition, RequisitionItem, Prefix, CurrentItemStock, PurchaseOrder, POItemDetails, IssueToClinic, IssueToClinicItems, GRN, GRNItem, IssueItems, Consumption, ReceiveAgainstIssue } = require('../models/mainInvSchema');
const { ItemMasterNew, ItemStoreTax, ItemConv, ItemSupplier } = require('../models/adminInventorySchema');
const { ClinicConfiguration } = require('../models/clinicConfig');
const {Op,Sequelize}= require('sequelize');
const {sequelize}=require('../sequelize')

const getClinicsForUser = async (userClinicId, role) => {
    if (role === 1) {
      // If the user is an admin, fetch all clinics under the main clinic
      const subClinics = await ClinicConfiguration.findAll({
        where: { clinic_id: userClinicId },
      });
  
      // Return the clinic ID for the main clinic plus all sub-clinic IDs
      return [userClinicId, ...subClinics.map(c => c.id)];
    }
  
    // If the user is a doctor, they only have access to their own clinic
    return [userClinicId];
  };

const createOpeningBalance = async (req, res) => {
    console.log(req.body);
    try {
        const { clinic, store, total_cgst_amount, total_sgst_amount, total_igst_amount, total_net_amount, total_amount, remarks, items } = req.body;

        // Extract batch codes from the items
        const batchCodes = items.map(item => item.batch_code);

        // Check if any of the batch codes already exist in the OpeningBalanceItem table
        const existingItems = await OpeningBalanceItem.findAll({
            where: {
                batch_code: batchCodes
            }
        });

        if (existingItems.length > 0) {
            // If any batch codes are found, return an error
            return res.status(400).json({
                success: false,
                message: `Batch codes already exist: ${existingItems.map(item => item.batch_code).join(', ')}`
            });
        }

        // Create opening balance record
        const openingBalance = await OpeningBalance.create({
            clinic_id:req.user.clinicId,
            clinic,
            store,
            total_cgst_amount,
            total_sgst_amount,
            total_igst_amount,
            total_net_amount,
            total_amount,
            remarks
        });

        // Create associated items and include clinic name in each item
        const openingBalanceItems = items.map(item => ({
            ...item,
            opening_balance_id: openingBalance.id,
            clinic: clinic,
            store: store,
            clinic_id:req.user.clinicId, // Add clinic and store to each item
        }));

        await OpeningBalanceItem.bulkCreate(openingBalanceItems);

        // Create a new record in CurrentItemStock for each item
        // const currentItemStockPromises = items.map(async (item) => {
        //     await CurrentItemStock.create({
        //         clinic: clinic,
        //         store: store,
        //         item_id: item.item_id,
        //         item_code: item.item_code,
        //         batchCode: item.batch_code,
        //         uom: item.uom,
        //         availableStock: item.total_quantity,
        //         expiryDate: item.expiry_date,
        //         isFree: false // or set based on your application logic
        //     });
        // });

        // await Promise.all(currentItemStockPromises);

        res.status(201).json({ success: true, message: 'Opening balance and items saved successfully' });
    } catch (error) {
        console.error('Error saving opening balance and items:', error);
        res.status(500).json({ success: false, message: 'Failed to save opening balance and items' });
    }
};



const getOpeningBalance = async (req, res) => {
    console.log(req.body);
    try {
        const op = await OpeningBalanceItem.findAll();
        res.status(200).json(op);
    } catch (error) {
        console.error('Error fetching opening balance details:', error);
        res.status(500).json({ msg: 'An error occurred while fetching opening balance details.' });
    }
};
const getCurrentItemStock = async (req, res) => {
    console.log(req.body);
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const op = await CurrentItemStock.findAll({
            where:{
                clinic_id:clinicIds
            }
        });
        res.status(200).json(op);
    } catch (error) {
        console.error('Error fetching store indents:', error);
        res.status(500).json({ msg: 'An error occurred while fetching store indents.' });
    }
}
const getCurrentItemStockByStore = async (req, res) => {
    console.log(req.query);
    try {
        const op = await CurrentItemStock.findAll({
            where:{clinic_id:req.user.clinicId,store_name:req.query.store}
        });
        res.status(200).json(op);
    } catch (error) {
        console.error('Error fetching store indents:', error);
        res.status(500).json({ msg: 'An error occurred while fetching store indents.' });
    }
}

const getStoreIndent = async (req, res) => {
    console.log(req.body);
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const op = await Indent.findAll({
            where:{
                clinic_id:clinicIds
            }
        });
        res.status(200).json(op);
    } catch (error) {
        console.error('Error fetching store indents:', error);
        res.status(500).json({ msg: 'An error occurred while fetching store indents.' });
    }
};
const getConsume = async (req, res) => {
    console.log(req.body);
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const data = await Consumption.findAll({
            where:{
                clinic_id:clinicIds
            }
        });
        res.status(200).json(data);
    } catch (error) {
        console.error('Error fetching store indents:', error);
        res.status(500).json({ msg: 'An error occurred while fetching store indents.' });
    }
};

const getItemDetails = async (req, res) => {
    const itemId = req.params.itemId;
    try {
        const itemDetails = await ItemMasterNew.findOne({ where: { id: itemId } });
        if (itemDetails) {
            console.log(itemDetails);
            res.status(200).json(itemDetails);
        } else {
            // Return an empty object if no item is found
            res.status(200).json({});
        }
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};



const getIndentDetails = async (req, res) => {
    const itemId = req.params.itemId;
    try {
        const contractDetails = await IndentItem.findAll({
            where: { indent_id: itemId }
        });

        res.status(200).json({ details: contractDetails });
    } catch (error) {
        console.error('Error fetching indent details:', error);
        res.status(500).json({ msg: 'An error occurred while fetching indent details.' });
    }
};

const getItems = async (req, res) => {
    try {
        const item_code = req.params.item_code;
        console.log(item_code);
        const item = await ItemMasterNew.findOne({
            where: { item_code: item_code}
        });

        if (!item) {
            return res.status(404).json({ message: 'Item not found' });
        }

        res.status(200).json(item);
    } catch (error) {
        console.error('Error fetching item details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const getFinancialYear = () => {
    const today = new Date();
    const year = today.getFullYear();
    const month = today.getMonth() + 1;
    let startYear, endYear;

    if (month >= 4) {
        startYear = year % 100;
        endYear = (year + 1) % 100;
    } else {
        startYear = (year - 1) % 100;
        endYear = year % 100;
    }

    return `${startYear}-${endYear}`;
};

const createOrUpdateIndent = async (req, res) => {
    console.log(req.body);
    try {
        const {
            clinic,
            indent_no,
            indent_date,
            exp_del_date,
            from_store,
            to_store,
            mr_no_pat,
            patient_name,
            remark,
            reference_no,
            po_no,
            grn_no,
            issue_no,
            status,
            freeze,
            approved,
            items,
            query ,// Variable to control update (set to 1 for updating)
        } = req.body;

        const financialYear = getFinancialYear();
        const clinicId = req.user.clinicId; 
        if (query === '1' && indent_no) {
            // Check if the indent exists
            const indent = await Indent.findOne({ where: { indent_no } });

            if (!indent) {
                return res.status(404).json({ message: 'Indent not found' });
            }

            // Update the indent fields
            await indent.update({
                clinic,
                indent_date,
                exp_del_date,
                from_store,
                to_store,
                mr_no_pat,
                patient_name,
                remark,
                reference_no,
                po_no,
                grn_no,
                issue_no,
                status,
                freeze,
                approved,
                items,
                clinic_id:clinicId // Save items as JSON
            });

            return res.status(200).json({ message: 'Indent updated successfully', indent });
        } else {
             // Check if `po_no` already exists
        if (indent_no) {
            const existingOrder = await Indent.findOne({ where: { indent_no } });
            if (existingOrder) {
                return res.status(400).json({ message: `Indent  ${indent_no} already exists.` });
            }
        }
            // Create a new indent
            const prefixRecord = await Prefix.findOne({ where: { name: 'Indent' } });
            if (!prefixRecord) {
                return res.status(404).json({ message: 'Prefix for Indent not found' });
            }
            const prefix = prefixRecord.prefix;

            // Create the new indent
            const indent = await Indent.create({
                clinic,
                indent_date,
                exp_del_date,
                from_store,
                to_store,
                mr_no_pat,
                patient_name,
                remark,
                reference_no,
                po_no,
                grn_no,
                issue_no,
                status,
                freeze,
                approved,
                items ,
                clinic_id:clinicId// Save items as JSON
            });

            // Generate the indent number
            const new_indent_no = indent_no || `${financialYear}/${prefix}/${indent.id}`;
            await indent.update({ indent_no: new_indent_no });

            return res.status(201).json({
                message: 'Indent created successfully',
                indent: { ...indent.toJSON(), indent_no: new_indent_no }
            });
        }
    } catch (error) {
        console.error('Error creating or updating indent:', error);
        res.status(500).json({ message: 'Error creating or updating indent', error });
    }
};

const createReceiveAgainstIssue = async (req, res) => {
    console.log(req.body);

    try {
        const {
            receive_no,
            receive_date,
            from_store,
            to_store,
            mr_no_pat,
            patient_name,
            remark,
            issueNo,
            items, // Array of items to save as JSON
        } = req.body;

        const financialYear = getFinancialYear(); // Helper function to get the current financial year
        const clinic_id = req.user.clinicId;

        // Validate if `receive_no` already exists
        if (receive_no) {
            const existingReceive = await ReceiveAgainstIssue.findOne({ where: { receive_no } });
            if (existingReceive) {
                return res.status(400).json({ message: `Receive record with receive_no ${receive_no} already exists.` });
            }
        }

        // Retrieve the prefix for `Receive` numbers
        const prefixRecord = await Prefix.findOne({ where: { name: 'Receive' } });
        if (!prefixRecord) {
            return res.status(404).json({ message: 'Prefix for Receive not found' });
        }
        const prefix = prefixRecord.prefix;

        // Create a new ReceiveAgainstIssue record
        const receiveRecord = await ReceiveAgainstIssue.create({
            clinic_id,
            receive_no: null, // Temporarily set to null; will be updated after generating the number
            receive_date,
            from_store,
            to_store,
            mr_no_pat,
            patient_name,
            remark,
            issueNo,
            items // Save items as JSON
        });

        // Generate the receive_no using financial year, prefix, and record ID
        const new_receive_no = receive_no || `${financialYear}/${prefix}/${receiveRecord.id}`;
        await receiveRecord.update({ receive_no: new_receive_no });

        // Update CurrentItemStock based on received items
        for (const item of items) {
            const {
                item_code,
                item_name,
                batch_code,
                uom,
                receive_quantity,
                cost_price,
                expiry_date,
                mrp,
                gst_percent
            } = item;

            // Fetch `item_group` and `item_category` from ItemMaster based on item_code
            const itemMasterRecord = await ItemMasterNew.findOne({
                where: { item_code },
                attributes: ['item_group', 'item_category'],
            });

            const item_group = itemMasterRecord?.item_group || '';
            const item_category = itemMasterRecord?.item_category || '';

            // Check if the item already exists in the current stock
            const existingStock = await CurrentItemStock.findOne({
                where: {clinic_id:req.user.clinicId, item_code, batch_code, store_name: to_store },
            });

            if (existingStock) {
                // Update the existing stock
                const newAvailableStock = existingStock.available_stock + receive_quantity;
                const newTotalMRP = (parseFloat(existingStock.total_mrp) || 0) + mrp * receive_quantity;
                const newTotalCP = (parseFloat(existingStock.total_cp) || 0) + cost_price * receive_quantity;

                await existingStock.update({
                    available_stock: newAvailableStock,
                    total_mrp: newTotalMRP,
                    total_cp: newTotalCP,
                    base_cp: cost_price,
                    base_mrp: mrp,
                });
            } else {
                // Insert a new stock record
                await CurrentItemStock.create({
                    clinic: clinic_id,
                    store_name: to_store,
                    item_code,
                    item_name,
                    gst_percent,
                    item_group,
                    item_category,
                    isFree: false,
                    batch_code,
                    available_stock: receive_quantity,
                    stocking_uom: uom,
                    expiry_date: expiry_date,
                    total_mrp: mrp * receive_quantity,
                    total_cp: cost_price * receive_quantity,
                    base_cp: cost_price,
                    base_mrp: mrp,
                });
            }
        }

        // Update the pending_quantity in issuedata based on received items
        const issueRecord = await IssueToClinic.findOne({ where: { issue_number: issueNo } });

        if (!issueRecord) {
            return res.status(404).json({ message: `Issue record with issueNo ${issueNo} not found.` });
        }

        // Parse the items from the issue record
        const issueItems = issueRecord.items;

        // Update pending_quantity for the items
        const updatedIssueItems = issueItems.map((issueItem) => {
            const receivedItem = items.find((item) => item.item_code === issueItem.item_code && item.batch_code === issueItem.batch_code);

            if (receivedItem) {
                // Decrease the pending quantity
                const newPendingQuantity = Math.max((issueItem.pending_quantity || 0) - receivedItem.receive_quantity, 0);
                return { ...issueItem, pending_quantity: newPendingQuantity };
            }

            return issueItem;
        });

        // Update the issue record with updated items
        await issueRecord.update({ items: updatedIssueItems });

        return res.status(201).json({
            message: 'Receive record created successfully, stock updated, and issue pending quantities updated',
            receiveRecord: { ...receiveRecord.toJSON(), receive_no: new_receive_no }
        });
    } catch (error) {
        console.error('Error creating ReceiveAgainstIssue:', error);
        res.status(500).json({ message: 'Error creating ReceiveAgainstIssue', error });
    }
};







const createOrUpdateRequisition = async (req, res) => {
    console.log(req.body);
    try {
        const clinicId = req.user.clinicId;  // Get clinic_id from session

    
        const { requisition_no, requisition_date, due_date, from_store, to_store, remark, freeze, items } = req.body;
        const financialYear = getFinancialYear();

        let requisition;
        if (requisition_no) {
            requisition = await Requisition.findOne({ where: { requisition_no } });
            if (!requisition) {
                return res.status(404).json({ message: 'Requisition not found' });
            }

            await Requisition.update({
                
                requisition_date,
                due_date,
                from_store,
                to_store,
                remark,
                freeze,
                items,
                clinic_id:clinicId
            }, { where: { requisition_no } });

            // await RequisitionItem.destroy({ where: { requisition_id: requisition.id } });

            // const requisitionItems = items.map(item => ({
            //     requisition_id: requisition.id,
            //     item_code: item.item_code,
            //     requisition_quantity: item.requisition_quantity,
            //     uom: item.uom,
            //     available_stock: item.available_stock
            // }));
            // await RequisitionItem.bulkCreate(requisitionItems);

            res.status(200).json({ message: 'Requisition updated successfully', requisition: { ...requisition.toJSON(), requisition_no } });
        } else {
            const prefixRecord = await Prefix.findOne({ where: { name: 'Purchase Requisition' } });
            if (!prefixRecord) {
                return res.status(404).json({ message: 'Prefix for Purchase Requisition not found' });
            }
            const prefix = prefixRecord.prefix;

            requisition = await Requisition.create({
                requisition_date,
                due_date,
                from_store,
                to_store,
                remark,
                freeze,
                items,
                clinic_id:clinicId
            });

            const new_requisition_no = `${financialYear}/${prefix}/${requisition.id}`;
            await Requisition.update({ requisition_no: new_requisition_no }, { where: { id: requisition.id } });

            // const requisitionItems = items.map(item => ({
            //     requisition_id: requisition.id,
            //     item_code: item.item_code,
            //     requisition_quantity: item.requisition_quantity,
            //     uom: item.uom,
            //     available_stock: item.available_stock
            // }));

            // await RequisitionItem.bulkCreate(requisitionItems);

            res.status(201).json({ message: 'Requisition created successfully', requisition: { ...requisition.toJSON(), requisition_no: new_requisition_no } });
        }
    } catch (error) {
        console.error('Error creating or updating requisition:', error);
        res.status(500).json({ message: 'Error creating or updating requisition', error });
    }
};

const deleteIndent = async (req, res) => {
    try {
        const indentId = req.params.id;
        const result = await Indent.destroy({
            where: { id: indentId }
        });

        if (result === 0) {
            return res.status(404).json({ message: 'Indent not found' });
        }

        res.status(200).json({ message: 'Indent deleted successfully' });
    } catch (error) {
        console.error('Error deleting indent:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const updateFreezeStatus = async (req, res) => {
    try {
        const { type, id } = req.params;
        const { freeze } = req.body;
  
        if (type === 'indent') {
            await Indent.update({ freeze }, { where: { id } });
        } else if (type === 'requisition') {
            await Requisition.update({ freeze }, { where: { id } });
        } else if (type === 'po') {
            await PurchaseOrder.update({ freeze }, { where: { id } });
        } else {
            return res.status(400).json({ message: 'Invalid type specified' });
        }
  
        res.status(200).json({ message: 'Freeze status updated successfully' });
    } catch (error) {
        console.error('Error updating freeze status:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
  };
  


const getIndent = async (req, res) => {
    const { id } = req.params;
    try {
        const indent = await Indent.findByPk(id);
        const items = await IndentItem.findAll({
            where: { indent_id: id }
        });

        const itemDetailsPromises = items.map(async (item) => {
            console.log(item)
            const itemDetails = await ItemMasterNew.findOne({
                where: { item_code: item.item_code },  // Use item_code to find the item details
                attributes: ['item_name', 'purchase_uom', 'stocking_uom']
            });
            return {
                ...item.toJSON(),
                itemDetails: itemDetails ? itemDetails.toJSON() : {}
            };
        });

        const detailedItems = await Promise.all(itemDetailsPromises);

        res.json({ indent, items: detailedItems });
    } catch (error) {
        console.error('Error fetching indent:', error);
        res.status(500).json({ message: 'Error fetching indent', error });
    }
};


const getAllRequisitions = async (req, res) => {
  try {
    const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
    const requisitions = await Requisition.findAll({
      where: {
        status: { [Op.ne]: 0 } ,
        clinic_id:clinicIds  
      }
    });
    res.status(200).json(requisitions);
  } catch (error) {
    console.error('Error fetching requisitions:', error);
    res.status(500).json({ message: 'An error occurred while fetching requisitions.' });
  }
};

const getRequisitionDetails = async (req, res) => {
  const { id } = req.params;
  try {
    const requisition = await Requisition.findByPk(id);
    if (!requisition) {
      return res.status(404).json({ message: 'Requisition not found' });
    }

    const items = await RequisitionItem.findAll({
      where: { requisition_id: id }
    });
    console.log(items);

    // Fetch item details from ItemMasterNew based on item_id
    const itemDetailsPromises = items.map(async (item) => {
      const itemDetails = await ItemMasterNew.findByPk(item.item_id, {
        attributes: ['item_name', 'purchase_uom', 'stocking_uom']
      });
      return {
        ...item.toJSON(),
        itemDetails: itemDetails ? itemDetails.toJSON() : {}
      };
    });

    const detailedItems = await Promise.all(itemDetailsPromises);

    res.json({ requisition, items: detailedItems });
  } catch (error) {
    console.error('Error fetching requisition:', error);
    res.status(500).json({ message: 'Error fetching requisition', error });
  }
};

const getRequisition = async (req, res) => {
  const { id } = req.params;
  try {
    const requisition = await Requisition.findByPk(id);
    if (!requisition) {
      return res.status(404).json({ message: 'Requisition not found' });
    }


    res.json({ requisition });
  } catch (error) {
    console.error('Error fetching requisition:', error);
    res.status(500).json({ message: 'Error fetching requisition', error });
  }
};

const deleteRecord = async (req, res) => {
    try {
        const { type, id } = req.params;

        let itemModel, mainModel, foreignKey;

        // Determine the models and foreign key based on the type
        if (type === 'requisition') {
            itemModel = RequisitionItem;
            mainModel = Requisition;
            foreignKey = 'requisition_id';
        } else if (type === 'indent') {
            itemModel = IndentItem;
            mainModel = Indent;
            foreignKey = 'indent_id';
        } else if (type === 'po') {
            itemModel = POItemDetails;
            mainModel = PurchaseOrder;
            foreignKey = 'purchase_order_id';
        } else {
            return res.status(400).json({ message: 'Invalid type specified' });
        }

        // Update the associated items' status to 0 (soft delete)
        await itemModel.update(
            { status: 0 },
            {
                where: { [foreignKey]: id }
            }
        );

        // Update the main record's status to 0 (soft delete)
        const result = await mainModel.update(
            { status: 0 },
            {
                where: { id }
            }
        );

        if (result[0] === 0) {
            return res.status(404).json({ message: `${type.charAt(0).toUpperCase() + type.slice(1)} not found` });
        }

        res.status(200).json({ message: `${type.charAt(0).toUpperCase() + type.slice(1)} status updated to 0 (soft deleted) successfully` });
    } catch (error) {
        console.error(`Error updating status for ${type}:`, error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

  

const getUniqueItemsWithStock = async (req, res) => {
    try {
        const { store, clinic } = req.query;
        const whereCondition = { store };

        if (clinic) {
            whereCondition.clinic = clinic;
        }

        console.log('Request Query:', req.query);

        // Step 1: Get unique item_ids with summed availableStock and uom based on store and clinic (if provided)
        const uniqueItems = await CurrentItemStock.findAll({
            attributes: [
                'item_id',
                'uom',
                [Sequelize.fn('SUM', Sequelize.col('availableStock')), 'totalAvailableStock']
            ],
            where: whereCondition,
            group: ['item_id', 'uom']
        });

        console.log(uniqueItems);

        const itemIds = uniqueItems.map(item => item.item_id);
        const stockMap = uniqueItems.reduce((map, item) => {
            const key = `${item.item_id}-${item.uom}`;
            map[key] = {
                totalAvailableStock: item.get('totalAvailableStock'),
                uom: item.uom
            };
            return map;
        }, {});

        // Step 2: Fetch item details from ItemMasterNew based on the unique item_ids
        const itemDetails = await ItemMasterNew.findAll({
            where: {
                id: {
                    [Op.in]: itemIds
                },
                clinic_id:req.user.clinicId
            }
        });

        // Extract itemCodes from the item details for tax and conversion lookup
        const itemCodes = itemDetails.map(item => item.item_code);

        // Step 3: Fetch tax details from ItemStoreTax based on itemCodes
        const taxDetails = await ItemStoreTax.findAll({
            where: {
                itemCode: {
                    [Op.in]: itemCodes
                }
            },
            attributes: ['itemCode', 'CGST', 'SGST', 'IGST']
        });

        // Create a map for quick lookup of tax details by itemCode
        const taxMap = taxDetails.reduce((map, tax) => {
            map[tax.itemCode] = tax;
            return map;
        }, {});

        // Step 4: Fetch all conversion details from ItemConv based on itemCodes
        const conversionDetails = await ItemConv.findAll({
            where: {
                itemCode: {
                    [Op.in]: itemCodes
                }
            }
        });

        // Create a map for quick lookup of conversion details by itemCode
        const conversionMap = conversionDetails.reduce((map, conv) => {
            if (!map[conv.itemCode]) {
                map[conv.itemCode] = [];
            }
            map[conv.itemCode].push(conv);
            return map;
        }, {});

        // Step 5: Combine the item details with the totalAvailableStock, uom, tax details, and conversion details
        const result = itemDetails.map(item => {
            const stockInfo = uniqueItems.find(ui => ui.item_id === item.id); // Find the corresponding uniqueItem
            const itemKey = `${item.id}-${stockInfo ? stockInfo.uom : ''}`;
            const stockData = stockMap[itemKey] || {};
            const tax = taxMap[item.item_code] || {};
            const conversions = conversionMap[item.item_code] || [];
            return {
                ...item.toJSON(),
                availableStock: stockData.totalAvailableStock || 0,
                uom: stockData.uom || '',
                CGST: tax.CGST || 0,
                SGST: tax.SGST || 0,
                IGST: tax.IGST || 0,
                conversions: conversions.map(conv => conv.toJSON())
            };
        });

        console.log(result);

        // Send the result as JSON
        return res.status(200).json({
            message: 'Items fetched successfully',
            data: result
        });
    } catch (error) {
        console.error('Error fetching unique items and their stock:', error);
        return res.status(500).json({
            message: 'Error fetching unique items and their stock',
            error: error.message
        });
    }
};

const savePurchaseOrder = async (req, res) => {
    console.log(req.body);
    const clinicId = req.user.clinicId; 
    try {
        const {
            po_no,
            date,
            store,
            payment_mode,
            supplier,
            payment_terms,
            delivery,
            delivery_duration,
            guarantee_warranty,
            instructions,
            gross_amount,
            total_gst,
            other_charges,
            po_discount,
            total_net_amount,
            remarks,
            pr_no,
            items // Keep items as is (array)
        } = req.body;

        // Get the financial year
        const financialYear = getFinancialYear();

        // Find the prefix for purchase order
        const prefixRecord = await Prefix.findOne({ where: { name: 'Purchase Order' } });
        if (!prefixRecord) {
            return res.status(404).json({ message: 'Prefix for Purchase Order not found' });
        }
        const prefix = prefixRecord.prefix;

        // Check if `po_no` already exists
        if (po_no) {
            const existingOrder = await PurchaseOrder.findOne({ where: { po_no } });
            if (existingOrder) {
                return res.status(400).json({ message: `Purchase Order number ${po_no} already exists.` });
            }
        }

        // Create the purchase order
        const purchaseOrder = await PurchaseOrder.create({
            date,
            store,
            payment_mode,
            supplier,
            payment_terms,
            delivery,
            delivery_duration,
            guarantee_warranty,
            instructions,
            gross_amount,
            total_gst,
            other_charges,
            po_discount,
            total_net_amount,
            remarks,
            pr_no,
            items: items,
            clinic_id:clinicId, // Save as JSON

        });

        // Generate and update the `po_no` if not provided
        const new_po_no = po_no || `${financialYear}/${prefix}/${purchaseOrder.id}`;
        await PurchaseOrder.update({ po_no: new_po_no }, { where: { id: purchaseOrder.id } });

        // If `pr_no` exists, update the `pr_pending_quantity` in the Requisition table
        if (pr_no) {
            // Find the Requisition record
            const requisition = await Requisition.findOne({ where: { requisition_no: pr_no } });
            if (!requisition) {
                return res.status(404).json({ message: 'Requisition not found' });
            }

            // Parse the items JSON from the Requisition
            let requisitionItems = requisition.items;

            // Iterate through the purchase order items and update the requisition items
            items.forEach((poItem) => {
                const { item_code, purchase_quantity } = poItem;

                // Find the matching item in the requisition items
                const reqItem = requisitionItems.find((item) => item.item_code === item_code);
                if (reqItem) {
                    // Update the `pr_pending_quantity`
                    const newPendingQuantity = Math.max(
                        parseFloat(reqItem.pr_pending_quantity || 0) - parseFloat(purchase_quantity || 0),
                        0
                    );
                    reqItem.pr_pending_quantity = newPendingQuantity.toString();
                }
            });

            // Update the Requisition record with the modified items
            await Requisition.update(
                { items: requisitionItems },
                { where: { requisition_no: pr_no } }
            );
        }

        // Send a success response with the `po_no`
        res.status(201).json({
            message: 'Purchase order and requisition updated successfully!',
            purchaseOrder: { ...purchaseOrder.toJSON(), po_no: new_po_no },
            po_no: new_po_no
        });

    } catch (error) {
        console.error('Error saving purchase order:', error);
        res.status(500).json({ message: 'An error occurred while saving the purchase order', error: error.message });
    }
};






const approvePR = async (req, res) => {
    console.log('Approve PR request received');
    try {
        const { id } = req.params;

        // Check if the requisition exists
        const requisition = await Requisition.findOne({ where: { id } });

        if (!requisition) {
            return res.status(404).json({ message: 'Requisition not found' });
        }

        // Check if the requisition is already approved
        if (requisition.approved) {
            return res.status(400).json({ message: 'Requisition is already approved' });
        }

        // Update the requisition to mark it as approved
        await Requisition.update(
            { approved: true }, // Assuming there is an 'approved' field in the Requisition model
            { where: { id } }
        );

        res.status(200).json({ message: 'Requisition approved successfully' });
    } catch (error) {
        console.error('Error approving requisition:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};


  const approveIndent = async (req, res) => {
    console.log('Request received for approval');
    try {
        const { id } = req.params;
        console.log(`Indent ID: ${id}`);
  
        // Check if the indent exists
        const indent = await Indent.findOne({ where: { id } });
  
        if (!indent) {
            console.log('Indent not found');
            return res.status(404).json({ message: 'Indent not found' });
        }

        // Check if the indent is already approved
        if (indent.approved) {
            console.log('Indent already approved');
            return res.status(400).json({ message: 'Indent has already been approved' });
        }
  
        // Update the indent to mark it as approved
        await Indent.update(
            { approved: true }, // Assuming there is an 'approved' field in the Indent model
            { where: { id } }
        );
  
        console.log('Indent approved successfully');
        res.status(200).json({ message: 'Indent approved successfully' });
    } catch (error) {
        console.error('Error approving indent:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const approvePO = async (req, res) => {
    console.log('Request received for approval');
    try {
        const { id } = req.params;

        // Check if the indent exists
        const indent = await PurchaseOrder.findOne({ where: { id } });

        if (!indent) {
            console.log('Indent not found');
            return res.status(404).json({ message: 'Indent not found' });
        }

        // Check if the indent is already approved
        if (indent.approved) {
            console.log('Indent is already approved');
            return res.status(400).json({ message: 'Indent is already approved' });
        }

        // Update the indent to mark it as approved
        await PurchaseOrder.update(
            { approved: true }, // Assuming there is an 'approved' field in the PurchaseOrder model
            { where: { id } }
        );

        console.log('Approved successfully');
        res.status(200).json({ message: 'Approved successfully' });
    } catch (error) {
        console.error('Error approving indent:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const approveGRN = async (req, res) => {
    console.log('Request received for GRN approval');
    try {
        const { id } = req.params;

        // Check if the GRN exists
        const grn = await GRN.findOne({ where: { id } });

        if (!grn) {
            console.log('GRN not found');
            return res.status(404).json({ message: 'GRN not found' });
        }

        // Check if the GRN is already approved
        if (grn.approved) {
            console.log('GRN is already approved');
            return res.status(400).json({ message: 'GRN is already approved' });
        }

        // Parse the items from the GRN
        const items = grn.items;

        // Iterate through the GRN items and update the CurrentItemStock
        for (const item of items) {
            const {
                item_code,
                item_name,
                batch_code,
                gst_percent,
                uom,
                received_quantity,
                cost_price,
                expiry_date,
                mrp,
                rack = '',
                shelf = '',
                bin = '',
                free_quantity = 0,
            } = item;

            // Fetch `item_group` and `item_category` from ItemMaster based on item_code
            const itemMasterRecord = await ItemMasterNew.findOne({
                where: { item_code },
                attributes: ['item_group', 'item_category'],
            });

            const item_group = itemMasterRecord?.item_group || '';
            const item_category = itemMasterRecord?.item_category || '';

            // Calculate total quantity (received + free)
            const total_quantity = received_quantity + parseInt(free_quantity, 10);

            // Check if the item already exists in the current stock
            const existingStock = await CurrentItemStock.findOne({
                where: { clinic_id:req.user.clinicId,item_code, batch_code, store_name: grn.store },
            });

            if (existingStock) {
                // Update the existing stock
                const newAvailableStock = existingStock.available_stock + total_quantity;
                const newTotalMRP = (parseFloat(existingStock.total_mrp) || 0) + mrp * received_quantity;
                const newTotalCP = (parseFloat(existingStock.total_cp) || 0) + cost_price * received_quantity;

                await existingStock.update({
                    available_stock: newAvailableStock,
                    total_mrp: newTotalMRP,
                    total_cp: newTotalCP,
                    base_cp: cost_price,
                    base_mrp: mrp,
                });
            } else {
                // Insert a new stock record
                await CurrentItemStock.create({
                    clinic_id:req.user.clinicId,
                    clinic: req.user.clinicName,
                    store_name: grn.store,
                    item_code,
                    item_name,
                    item_group,
                    item_category,
                    gst_percent,
                    isFree: false,
                    batch_code,
                    available_stock: total_quantity,
                    stocking_uom: uom,
                    expiry_date: expiry_date,
                    total_mrp: mrp * received_quantity, // Only include the cost of received items, not free items
                    total_cp: cost_price * received_quantity, // Only include the cost of received items, not free items
                    base_cp: cost_price,
                    base_mrp: mrp,
                    rack,
                    shelf,
                    bin,
                });
            }
        }

        // Update the GRN to mark it as approved
        await grn.update({ approved: true });

        console.log('GRN approved and stock updated successfully');
        res.status(200).json({ message: 'GRN approved and stock updated successfully' });
    } catch (error) {
        console.error('Error approving GRN:', error);
        res.status(500).json({ message: 'Internal server error', error: error.message });
    }
};




const fetchRequisitionItemsWithDetails = async (req, res) => {
    
    console.log('232323')
    console.log('sd',req.query)
    try {
        const store = req.query.store;

        // Step 1: Fetch all Requisition records where status is not 0, filtered by `to_store` and approved
        const storeRequisitions = await Requisition.findAll({
            where: {clinic_id:req.user.clinicId,status: { [Op.ne]: 0 }, to_store: store, approved: 1, },
        });

        // Check if no requisitions were found
        if (!storeRequisitions || storeRequisitions.length === 0) {
            return res.status(404).json({ message: `No requisitions found for store: ${store}` });
        }

        // Step 2: Iterate over each requisition and fetch item details along with current stock
        const results = await Promise.all(
            storeRequisitions.map(async (requisition) => {
                const items = requisition.items || []; // Directly access the items array

                // Fetch item details for each item in the requisition
                const detailedItems = await Promise.all(
                    items.map(async (item) => {
                        // Fetch item details from ItemMasterNew
                        const itemMaster = await ItemMasterNew.findOne({
                            where: { item_code: item.item_code },
                        });

                        // Fetch current stock for the item from CurrentItemStock
                        const stockData = await CurrentItemStock.findAll({
                            where: { item_code: item.item_code, store_name: store },
                            attributes: [
                                'item_code',
                                [sequelize.fn('SUM', sequelize.col('available_stock')), 'total_stock']
                            ],
                            group: ['item_code']
                        });

                        const totalStock = stockData.length > 0
                            ? parseInt(stockData[0].dataValues.total_stock, 10)
                            : 0;

                        // Construct item details object
                        const itemDetail = {
                            item_code: item.item_code,
                            item_name: item.item_name,
                            requisition_quantity: item.requisition_quantity,
                            pr_pending_quantity: item.pr_pending_quantity,
                            uom: item.uom,
                            available_stock: totalStock, // Add aggregated stock
                            itemMaster: itemMaster ? itemMaster.toJSON() : null,
                        };

                        return itemDetail;
                    })
                );

                // Construct the requisition object with its detailed items
                return {
                    requisition_no: requisition.requisition_no,
                    requisition_date: requisition.requisition_date,
                    from_store: requisition.from_store,
                    to_store: requisition.to_store,
                    items: detailedItems,
                };
            })
        );

        // Check if there are no items in any of the requisitions
        if (results.length === 0 || results.every(req => req.items.length === 0)) {
            return res.status(404).json({ message: 'Requisitions found, but no items available.' });
        }

        // Send the combined data as response
        res.status(200).json(results);
    } catch (error) {
        console.error('Error fetching requisition items with details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const fetchIssueItemsWithDetails = async (req, res) => {
    try {
        const store = req.query.store;

        // Step 1: Fetch all issue records where status is not 0, filtered by `from_store`
        const storeIssues = await IssueToClinic.findAll({
            where: {clinic_id:req.user.clinicId, from_store: store },
        });

        res.status(200).json(storeIssues);
    } catch (error) {
        console.error('Error fetching issue items with details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};


const fetchIndentItemsWithDetails = async (req, res) => {
    try {
        const store = req.query.store;

        // Step 1: Fetch all Indent records filtered by `to_store` and approved status
        const indents = await Indent.findAll({
            where: {
                clinic_id:req.user.clinicId,
                to_store: store,
                status: true,
                approved: true,
            },
        });

        // Check if no indents were found
        if (!indents || indents.length === 0) {
            return res.status(404).json({ message: `No indents found for store: ${store}` });
        }

        // Step 2: Iterate over each indent and fetch item details along with current stock
        const results = await Promise.all(
            indents.map(async (indent) => {
                const items = indent.items || []; // Directly access the items array

                // Fetch item details for each item in the indent
                const detailedItems = await Promise.all(
                    items.map(async (item) => {
                        // Fetch item details from ItemMasterNew
                        const itemMaster = await ItemMasterNew.findOne({
                            where: { item_code: item.item_code },
                        });

                        // Fetch current stock for the item from CurrentItemStock
                        const stockData = await CurrentItemStock.findAll({
                            where: { item_code: item.item_code, store_name: store },
                            attributes: [
                                'item_code',
                                [sequelize.fn('SUM', sequelize.col('available_stock')), 'total_stock']
                            ],
                            group: ['item_code']
                        });

                        const totalStock = stockData.length > 0
                            ? parseInt(stockData[0].dataValues.total_stock, 10)
                            : 0;

                        // Construct item details object
                        const itemDetail = {
                            item_code: item.item_code,
                            item_name: item.item_name,
                            indent_quantity: item.indent_quantity,
                            pending_quantity: item.pending_quantity,
                            uom: item.uom,
                            available_stock: totalStock, // Add aggregated stock
                            itemMaster: itemMaster ? itemMaster.toJSON() : null,
                        };

                        return itemDetail;
                    })
                );

                // Construct the indent object with its detailed items
                return {
                    indent_no: indent.indent_no,
                    indent_date: indent.indent_date,
                    exp_del_date: indent.exp_del_date,
                    from_store: indent.from_store,
                    to_store: indent.to_store,
                    remark: indent.remark,
                    items: detailedItems,
                };
            })
        );

        // Check if there are no items in any of the indents
        if (results.length === 0 || results.every(indent => indent.items.length === 0)) {
            return res.status(404).json({ message: 'Indents found, but no items available.' });
        }

        // Send the combined data as response
        res.status(200).json(results);
    } catch (error) {
        console.error('Error fetching indent items with details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};




const fetchPurchaseOrderItemsWithDetails = async (req, res) => {
    console.log(req.query);
    const { supplier} = req.query;

    try {
        // Step 1: Fetch all Purchase Orders
        const purchaseOrders = await PurchaseOrder.findAll({
            where:{clinic_id:req.user.clinicId,status: { [Op.ne]: 0 },supplier:supplier,approved:true}
        });
        console.log('Purchase Orders:', purchaseOrders);

       

        // Send the combined data as response
        res.status(200).json(purchaseOrders);
    } catch (error) {
        console.error('Error fetching purchase order items with details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};





// Function to save Issue and associated Items

  



  const getIssueToclinic = async (req, res) => {
    console.log('1')
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const item = await IssueToClinic.findAll({
            where:{
                clinic_id:clinicIds
            }
        });

        res.status(200).json(item);
    } catch (error) {
        console.error('Error fetching item details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};



const getAllPO = async (req, res) => {
   
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const item = await PurchaseOrder.findAll({
            where:{status: { [Op.ne]: 0 },clinic_id:clinicIds}
        });

        res.status(200).json(item);
    } catch (error) {
        console.error('Error fetching item details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};
const getAllGRN = async (req, res) => {
   
    try {
        const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
        const item = await GRN.findAll({
            where:{clinic_id:clinicIds,status: { [Op.ne]: 0 }}
        });

        res.status(200).json(item);
    } catch (error) {
        console.error('Error fetching item details:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

getPOItems = async (req, res) => {
    const poId = req.params.id;
    try {
        const poItems = await POItemDetails.findAll({
            where: { purchase_order_id: poId }
        });
        res.status(200).json(poItems);
    } catch (error) {
        console.error('Error fetching purchase order items:', error);
        res.status(500).json({ message: 'Error fetching purchase order items' });
    }
};

getIssueItems = async (req, res) => {
    const Id = req.params.id;
    try {
        const poItems = await IssueItems.findAll({
            where: { IssueToClinicId: Id }
        });
        res.status(200).json(poItems);
    } catch (error) {
        console.error('Error fetching purchase order items:', error);
        res.status(500).json({ message: 'Error fetching purchase order items' });
    }
};

const getItemsOnSearch = async (req, res) => {
    console.log(req.query);
    try {
        const store = req.query.store;
     
        console.log(store)
        const searchCriteria = {clinic_id:req.user.clinicId};

        // Build the search criteria from the query parameters
        if (req.query.item_name) searchCriteria.item_name = { [Op.like]: `%${req.query.item_name}%` };
        if (req.query.brand_name) searchCriteria.brand_name = { [Op.like]: `%${req.query.brand_name}%` };
        if (req.query.item_code) searchCriteria.item_code = req.query.item_code;
        if (req.query.item_category) searchCriteria.item_category = req.query.item_category;
        if (req.query.item_group) searchCriteria.item_group = req.query.item_group;
        if (req.query.molecule_name) searchCriteria.molecule_name = req.query.molecule_name;

        // Query the database for items using the search criteria
        const items = await ItemMasterNew.findAll({ where: searchCriteria });

        // If items are found, aggregate the stock information
        const itemCodes = items.map(item => item.item_code);
        const stockData = await CurrentItemStock.findAll({
            where: { item_code: { [Op.in]: itemCodes },store_name:store },
            attributes: [
                'item_code',
                [sequelize.fn('SUM', sequelize.col('available_stock')), 'total_stock']
            ],
            group: ['item_code']
        });

        // Map stock data for quick lookup
        const stockMap = stockData.reduce((acc, stock) => {
            acc[stock.item_code] = parseInt(stock.dataValues.total_stock, 10) || 0;
            return acc;
        }, {});

        // Combine item data with their available stock
        const combinedResults = items.map(item => {
            return {
                ...item.toJSON(),
                available_stock: stockMap[item.item_code] || 0 // Default to 0 if no stock found
            };
        });

        // Return the combined results
        res.status(200).json(combinedResults);
    } catch (error) {
        console.error('Error during search:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};


const getCurrentItemsOnSearch = async (req, res) => {
    console.log(req.query);
    try {
        
        // Build the search criteria for ItemMasterNew dynamically
        const itemSearchCriteria = {};

        if (req.query.item_name) itemSearchCriteria.item_name = { [Op.like]: `%${req.query.item_name}%` };
        if (req.query.item_code) itemSearchCriteria.item_code = { [Op.like]: `%${req.query.item_code}%` };
        if (req.query.item_category) itemSearchCriteria.item_category = { [Op.like]: `%${req.query.item_category}%` };
        if (req.query.item_group) itemSearchCriteria.item_group = { [Op.like]: `%${req.query.item_group}%` };
        if (req.query.brand_name) itemSearchCriteria.brand_name = { [Op.like]: `%${req.query.brand_name}%` };
        if (req.query.molecule_name) itemSearchCriteria.molecule_name = { [Op.like]: `%${req.query.molecule_name}%` };

        // Query ItemMasterNew with the search criteria
        const items = await ItemMasterNew.findAll({ where: itemSearchCriteria ,clinic_id:req.user.clinicId});

        if (items.length === 0) {
            return res.status(404).json({ message: 'No items found' });
        }

        // Extract item codes from the ItemMasterNew results
        const itemCodes = items.map(item => item.item_code);

        // Build the search criteria for CurrentItemStock using item codes and optional filters
        const stockSearchCriteria = { item_code: { [Op.in]: itemCodes } };

        if (req.query.batch_code) stockSearchCriteria.batchCode = req.query.batch_code; // Filter by batch code
        if (req.query.store) stockSearchCriteria.store_name = req.query.store; // Filter by store

        // Query CurrentItemStock based on the criteria
        const currentStocks = await CurrentItemStock.findAll({ where: stockSearchCriteria });

        if (currentStocks.length === 0) {
            return res.status(404).json({ message: 'No stock found for the given criteria' });
        }

        // Concurrently fetch related store tax details and conversion factors for the item codes
        const [storeTaxes, conversionFactors] = await Promise.all([
            ItemStoreTax.findAll({ where: { itemCode: { [Op.in]: itemCodes } } }),
            ItemConv.findAll({ where: { itemCode: { [Op.in]: itemCodes } } })
        ]);

        // Map the related data for easier access
        const itemDetailsMap = items.reduce((acc, item) => {
            acc[item.item_code] = item.toJSON();
            return acc;
        }, {});

        const storeTaxMap = storeTaxes.reduce((acc, tax) => {
            if (!acc[tax.itemCode]) acc[tax.itemCode] = [];
            acc[tax.itemCode].push(tax.toJSON());
            return acc;
        }, {});

        const conversionFactorMap = conversionFactors.reduce((acc, conv) => {
            if (!acc[conv.itemCode]) acc[conv.itemCode] = [];
            acc[conv.itemCode].push(conv.toJSON());
            return acc;
        }, {});

        // Combine the results
        const results = currentStocks.map(stock => {
            const itemCode = stock.item_code;
            return {
                ...stock.toJSON(),
                itemDetails: itemDetailsMap[itemCode] || {},
                storeTaxes: storeTaxMap[itemCode] || [],
                conversionFactors: conversionFactorMap[itemCode] || []
            };
        });

        // Respond with the combined results
        res.status(200).json(results);
    } catch (error) {
        console.error('Error during search:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};




const getItemsFromIndents = async (req, res) => {
    try {
        const searchCriteria = {};

        // Build the search criteria from the query parameters
        if (req.query.item_name) searchCriteria.item_name = { [Op.like]: `%${req.query.item_name}%` };
        if (req.query.indent_no) searchCriteria.indent_no = { [Op.like]: `%${req.query.indent_no}%` };
        if (req.query.item_code) searchCriteria.item_code = req.query.item_code;
        if (req.query.store) searchCriteria.store = req.query.store;

        // Query the Indent schema based on search criteria
        const indents = await Indent.findAll({
            where: req.query.indent_no // Use searchCriteria for querying indents
        });

        if (indents.length === 0) {
            return res.status(404).json({ message: 'No indents found' });
        }

        // Extract indent IDs and corresponding details for further queries
        const indentIds = indents.map(indent => indent.id);
        const indentDetails = indents.reduce((acc, indent) => {
            acc[indent.id] = {
                indent_no: indent.indent_no,
                indent_date: indent.indent_date
            };
            return acc;
        }, {});

        // Fetch items from IndentItems using the indent IDs
        const indentItems = await IndentItem.findAll({
            where: { indent_id: { [Op.in]: indentIds } }
        });

        if (indentItems.length === 0) {
            return res.status(404).json({ message: 'No items found for the specified indents' });
        }

        // Extract item codes from the retrieved indent items for further queries
        const itemCodes = indentItems.map(item => item.item_code);

        // Concurrently fetch store tax details, current item stock, conversion details, and item details using the item codes
        const [storeTaxes, currentStocks, conversionFactors, itemDetails] = await Promise.all([
            ItemStoreTax.findAll({
                where: { itemCode: { [Op.in]: itemCodes } }
            }),
            CurrentItemStock.findAll({
                where: { item_code: { [Op.in]: itemCodes }, store: req.query.store || '' }
            }),
            ItemConv.findAll({
                where: { itemCode: { [Op.in]: itemCodes } }
            }),
            ItemMasterNew.findAll({
                where: { item_code: { [Op.in]: itemCodes } ,clinic_id:req.user.clinicId }
            })
        ]);

        // Combine the indent items with their corresponding store tax details, current item stock, conversion factors, item details, and indent details
        const results = indentItems.map(item => {
            const itemTaxDetails = storeTaxes.filter(tax => tax.itemCode === item.item_code);
            const itemStockDetails = currentStocks.filter(stock => stock.item_code === item.item_code);
            const itemConversionDetails = conversionFactors.filter(conv => conv.itemCode === item.item_code);
            const itemDetail = itemDetails.find(detail => detail.item_code === item.item_code);
            const indentDetail = indentDetails[item.indent_id];

            return {
                ...item.toJSON(),
                indent_no: indentDetail.indent_no,
                indent_date: indentDetail.indent_date,
                storeTaxes: itemTaxDetails,
                currentStock: itemStockDetails,
                conversionFactors: itemConversionDetails,
                itemDetails: itemDetail
            };
        });

        // Return the combined results
        res.status(200).json(results);
    } catch (error) {
        console.error('Error during search:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};

const getItemsFromPR = async (req, res) => {
    console.log(req.query.from_store)
    try {
      

        // Query the Requisition schema based on search criteria
        const requisitions = await Requisition.findAll();

        if (requisitions.length === 0) {
            return res.status(404).json({ message: 'No requisitions found' });
        }

        // Extract requisition IDs and corresponding details for further queries
        const requisitionIds = requisitions.map(requisition => requisition.id);
        const requisitionDetails = requisitions.reduce((acc, requisition) => {
            acc[requisition.id] = {
                requisition_no: requisition.requisition_no,
                requisition_date: requisition.requisition_date
            };
            return acc;
        }, {});

        // Fetch items from RequisitionItem using the requisition IDs
        const requisitionItems = await RequisitionItem.findAll({
            where: { requisition_id: { [Op.in]: requisitionIds } }
        });

        if (requisitionItems.length === 0) {
            return res.status(404).json({ message: 'No items found for the specified requisitions' });
        }

        // Extract item codes from the retrieved requisition items for further queries
        const itemCodes = requisitionItems.map(item => item.item_code);

        // Concurrently fetch store tax details, current item stock, conversion details, and item details using the item codes
        const [storeTaxes, currentStocks, conversionFactors, itemDetails] = await Promise.all([
            ItemStoreTax.findAll({
                where: { itemCode: { [Op.in]: itemCodes } }
            }),
            CurrentItemStock.findAll({
                where: { item_code: { [Op.in]: itemCodes }, store: req.query.from_store || '' }
            }),
            ItemConv.findAll({
                where: { itemCode: { [Op.in]: itemCodes } }
            }),
            ItemMasterNew.findAll({
                where: { item_code: { [Op.in]: itemCodes },clinic_id:req.user.clinicId }
            })
        ]);

        // Combine the requisition items with their corresponding store tax details, current item stock, conversion factors, item details, and requisition details
        const results = requisitionItems.map(item => {
            const itemTaxDetails = storeTaxes.filter(tax => tax.itemCode === item.item_code);
            const itemStockDetails = currentStocks.filter(stock => stock.item_code === item.item_code);
            const itemConversionDetails = conversionFactors.filter(conv => conv.itemCode === item.item_code);
            const itemDetail = itemDetails.find(detail => detail.item_code === item.item_code);
            const requisitionDetail = requisitionDetails[item.requisition_id];

            return {
                ...item.toJSON(),
                requisition_no: requisitionDetail.requisition_no,
                requisition_date: requisitionDetail.requisition_date,
                storeTaxes: itemTaxDetails,
                currentStock: itemStockDetails,
                conversionFactors: itemConversionDetails,
                itemDetails: itemDetail
            };
        });

        // Return the combined results
        res.status(200).json(results);
    } catch (error) {
        console.error('Error during search:', error);
        res.status(500).json({ message: 'Internal server error' });
    }
};







getItemBatches = async (req, res) => {
    const Id = req.params.id;
    try {
        const Items = await CurrentItemStock.findAll({
            where: { item_id: Id }
        });
        console.log(Items)
        res.status(200).json(Items);
    } catch (error) {
        console.error('Error fetching purchase order items:', error);
        res.status(500).json({ message: 'Error fetching purchase order items' });
    }
};

const saveGRNData = async (req, res) => {
    console.log(req.body);
    const {
        grn_no,
        poNo,
        date,
        gate_entry_no,
        store,
        supplier,
        pay_mode,
        invoice_date,
        invoice_no,
        total_amount,
        gst_amount,
        other_charges,
        discount_amount,
        net_amount,
        received_by,
        remark,
        is_finalize,
        items
    } = req.body;

    const clinicId = req.user.clinicId; 
    try {
        // Check if a GRN record with the same `grn_no` already exists
        const existingGRN = await GRN.findOne({ where: { grn_no } });
        if (existingGRN) {
            return res.status(409).json({
                message: `A GRN with the number ${grn_no} already exists.`,
            });
        }

        // Retrieve the prefix for GRN records
        const prefixRecord = await Prefix.findOne({ where: { name: 'GRN' } });
        if (!prefixRecord) {
            return res.status(404).json({ message: 'Prefix for GRN not found' });
        }
        const prefix = prefixRecord.prefix;

        // Create the GRN record
        const grn = await GRN.create({
            poNo,
            date,
            gate_entry_no,
            store,
            supplier,
            pay_mode,
            invoice_date,
            invoice_no,
            total_amount,
            gst_amount,
            other_charges,
            discount_amount,
            net_amount,
            received_by,
            remark,
            is_finalize,
            items,
            clinic_id:clinicId // Store the entire items array in the JSON field
        });

        // Generate the new GRN number using a format
        const financialYear = getFinancialYear(date);
        const new_grn_no = `${financialYear}/${prefix}/${grn.id}`;

        // Update the GRN record with the new number
        await grn.update({ grn_no: new_grn_no });

        // If `poNo` exists, attempt to update the corresponding Purchase Order
        if (poNo) {
            const purchaseOrder = await PurchaseOrder.findOne({ where: { po_no: poNo } });
            if (purchaseOrder) {
                // Parse the items from the Purchase Order
                let poItems = purchaseOrder.items;

                // Iterate through GRN items to update `po_pending_quantity` in the Purchase Order items
                items.forEach((grnItem) => {
                    const { item_code, received_quantity } = grnItem;

                    // Find the matching item in the Purchase Order
                    const poItem = poItems.find((item) => item.item_code === item_code);
                    if (poItem) {
                        // Update the `po_pending_quantity`
                        const newPendingQuantity = Math.max(
                            parseFloat(poItem.po_pending_quantity || 0) - parseFloat(received_quantity || 0),
                            0
                        );
                        poItem.po_pending_quantity = newPendingQuantity.toString();
                    }
                });

                // Update the Purchase Order record with the modified items
                await PurchaseOrder.update(
                    { items: poItems },
                    { where: { po_no: poNo } }
                );
            } else {
                console.log(`Purchase Order with PO No: ${poNo} not found. Skipping PO update.`);
            }
        }

        res.status(201).send({
            message: 'GRN saved successfully!',
            grnId: grn.id,
            grnNo: new_grn_no,
        });
    } catch (error) {
        console.error('Failed to save GRN:', error);
        res.status(500).send({
            message: 'Failed to save GRN',
            error: error.message,
        });
    }
};




const saveIssueToClinic = async (req, res) => {
    console.log(req.body);
    const {
        issue_number,
        issue_date,
        from_store,
        to_store,
        is_quarantine,
        remark,
        total_cost_amount,
        mr_no,
        patient_name,
        pr_no,
        indent_no,
        items // This is the JSON array
    } = req.body;

    const transaction = await sequelize.transaction(); // Start a transaction for atomicity

    try {
        // Check for duplicate issue_number
        if (issue_number) {
            const existingIssue = await IssueToClinic.findOne({ where: { issue_number } });
            if (existingIssue) {
                return res.status(400).json({
                    message: `An issue with issue_number "${issue_number}" already exists. Duplicate issue_number is not allowed.`
                });
            }
        }

        // Retrieve the prefix for Issue to Clinic records
        const prefixRecord = await Prefix.findOne({ where: { name: 'Issue To Clinic' } });
        if (!prefixRecord) {
            return res.status(404).json({ message: 'Prefix for Issue To Clinic not found' });
        }
        const prefix = prefixRecord.prefix;

        // Create the Issue to Clinic record without the issue_number first
        const issue = await IssueToClinic.create({
            clinic_id:req.user.clinicId,
            issue_date,
            from_store,
            to_store,
            is_quarantine,
            remark,
            total_cost_amount,
            mr_no,
            patient_name,
            pr_no,
            indent_no,
            items: items // Save items array as a JSON string
        }, { transaction });

        // Generate the new issue number using a format that includes the issue ID
        const financialYear = getFinancialYear(issue_date); // Assuming a function to determine the financial year
        const newIssueNumber = `${financialYear}/${prefix}/${issue.id}`;

        // Check for duplicate issue_number in case of high concurrency
        const duplicateCheck = await IssueToClinic.findOne({ where: { issue_number: newIssueNumber } });
        if (duplicateCheck) {
            await transaction.rollback();
            return res.status(400).json({
                message: `An issue with issue_number "${newIssueNumber}" already exists. Duplicate issue_number is not allowed.`
            });
        }

        // Update the issue_number
        await issue.update({ issue_number: newIssueNumber }, { transaction });

        // **Update pending quantities for indent or PR**
        if (indent_no) {
            const indent = await Indent.findOne({ where: { indent_no }, transaction });
            if (!indent) {
                throw new Error(`Indent with indent_no "${indent_no}" not found.`);
            }

            const updatedItems = indent.items.map(item => {
                const issuedItem = items.find(i => i.item_code === item.item_code);
                if (issuedItem) {
                    const newPendingQuantity = Math.max(0, item.pending_quantity - issuedItem.issued_quantity);
                    return { ...item, pending_quantity: newPendingQuantity };
                }
                return item;
            });

            await indent.update({ items: updatedItems }, { transaction });
        }

        if (pr_no) {
            const pr = await Requisition.findOne({ where: { requisition_no: pr_no }, transaction });
            if (!pr) {
                throw new Error(`Requisition with requisition_no "${pr_no}" not found.`);
            }

            const updatedItems = pr.items.map(item => {
                const issuedItem = items.find(i => i.item_code === item.item_code);
                if (issuedItem) {
                    const newPendingQuantity = Math.max(0, item.pr_pending_quantity - issuedItem.issued_quantity);
                    return { ...item, pr_pending_quantity: newPendingQuantity };
                }
                return item;
            });

            await pr.update({ items: updatedItems }, { transaction });
        }

        // **Update stock quantities batch-wise**
        for (const item of items) {
            const { item_code, batch_code, issued_quantity } = item;

            // Subtract from `from_store`
            const fromStock = await CurrentItemStock.findOne({
                where: { clinic_id:req.user.clinicId,store_name: from_store, item_code, batch_code },
                transaction
            });

            if (!fromStock || fromStock.available_stock < issued_quantity) {
                throw new Error(`Insufficient stock for item_code "${item_code}" in batch "${batch_code}" at "${from_store}".`);
            }

            await fromStock.update(
                { available_stock: fromStock.available_stock - issued_quantity },
                { transaction }
            );
        }

        // Commit the transaction after all updates
        await transaction.commit();

        res.status(201).send({
            message: 'Issue to Clinic saved successfully!',
            issueId: issue.id,
            issueNumber: newIssueNumber
        });
    } catch (error) {
        console.error('Failed to save Issue to Clinic:', error);
        await transaction.rollback(); // Rollback transaction in case of errors
        res.status(500).send({
            message: 'Failed to save Issue to Clinic',
            error: error.message
        });
    }
};





const saveConsumption = async (req, res) => {
    console.log(req.user)
    console.log(req.session)
    console.log(req.body)
    try {
        const { store, from_date, to_date, item_list } = req.body;

        
        // Prepare the consumption data to be saved (initially without consumption_no)
        const consumptionData = {
            clinic_id:req.user.clinicId,
            store,
            from_date,
            to_date,
            item_list: item_list, // Save item_list as JSON in the database
            created_by: req.user ? req.user.username : 'Sahil', // If using authentication, add created_by
            created_on: new Date(),
            status: '' // Default status
        };

        // Save the consumption data to the database first
        const savedConsumption = await Consumption.create(consumptionData);

        // Generate the consumption number (CONS_ID) based on the saved record's ID
        const consumption_no = `CONS_${savedConsumption.id}`; // Using the ID from the saved record

        // Update the saved consumption with the generated consumption_no
        await savedConsumption.update({ consumption_no });

        // Send success response
        res.status(201).json({
            success: true,
            message: 'Consumption saved successfully',
            consumption_no: savedConsumption.consumption_no
        });
    } catch (error) {
        console.error('Error saving consumption:', error);
        res.status(500).json({
            success: false,
            message: 'Error saving consumption',
            error: error.message
        });
    }
};



const getBatchDetails = async (req, res) => {
    try {
        const { item_code, store } = req.query;

        if (!item_code || !store) {
            return res.status(400).json({ message: 'Item code and store are required' });
        }

        // Fetch batch details for the given item code and store
        const batchDetails = await CurrentItemStock.findAll({
            where: {
                item_code,
                store_name: store,
            },
            attributes: [
                'batch_code',
                'available_stock',
                'stocking_uom',
                'base_cp',
                'base_mrp',
                'expiry_date',
                'gst_percent'
            ],
        });

        if (!batchDetails || batchDetails.length === 0) {
            return res.status(404).json({ message: 'No batch details found for the given item and store' });
        }

        // Respond with batch details
        res.status(200).json(batchDetails);
    } catch (error) {
        console.error('Error fetching batch details:', error);
        res.status(500).json({ message: 'Internal server error', error: error.message });
    }
};



module.exports = {
    saveIssueToClinic,
    createOpeningBalance,
    getOpeningBalance,
    getItemDetails,
    createOrUpdateIndent,
    getStoreIndent,
    getIndentDetails,
    getItems,
    deleteIndent,
    updateFreezeStatus,
    getIndent,
    createOrUpdateRequisition,
    getAllRequisitions,
    getRequisitionDetails,
    getRequisition,
    getCurrentItemStock,
    getUniqueItemsWithStock,
    savePurchaseOrder,
    approvePR,
    fetchRequisitionItemsWithDetails,
    getIssueToclinic,
    getAllPO,
    getPOItems,
    deleteRecord,
    getItemsOnSearch,
    getItemBatches,
    saveGRNData,
    approveIndent,
    approvePO,
    getItemsFromIndents,
    getItemsFromPR,
    getIssueItems,
    fetchPurchaseOrderItemsWithDetails,
    saveConsumption,
    getConsume,
    getCurrentItemsOnSearch,
    getAllGRN,
    approveGRN,
    getBatchDetails,
    fetchIndentItemsWithDetails,
    getCurrentItemStockByStore,
    fetchIssueItemsWithDetails,
    createReceiveAgainstIssue
};
