const {
  PR_patientReg,
  PR_ReferralDoc,
  PR_Appointment,
  PR_PatientVisit,
  PR_formNewCouple,
  PR_BillFindPatient,
  BillServices,
  PR_BillPaymentDetails,
  PR_BillPharmacy,
  BillItems,
  Advance,
  Patients,
} = require("../models/PatientReg");
const session = require("express-session");
const { Op, where } = require("sequelize");
const { sequelize } = require("../sequelize");

const CryptoJS = require("crypto-js");
const {KYC}=require("../models/Kyc")

//decryption and encryption fxn
function decryptData(encryptedData, secretKey) {
  const bytes = CryptoJS.AES.decrypt(encryptedData, secretKey);
  return bytes.toString(CryptoJS.enc.Utf8);
}
function encryptDataForUrl(data) {
  console.log(data);
  const secretKey = "ll"; // Replace with your actual secret key
  const encrypted = CryptoJS.AES.encrypt(data, secretKey).toString();
  const encodedEncrypted = encodeURIComponent(encrypted);
  return encodedEncrypted;
}

const patientRagistration = async (req, res) => {
  try {
    // Extract the patient data directly from the request body

    const patientData = req.body;
    const file = req.files;
    const lead_no = req.body.lead_no; 

    // Check if no patient data is provided
    if (Object.keys(patientData).length === 0) {
      return res.status(400).json({ msg: "No patient data provided." });
    }

    // Parse patient data if sent as JSON string
    let pdata = JSON.parse(patientData.jsonString);
     pdata.lead_no = lead_no; 

    pdata.clinic_id = req.user.clinicId;
    // Validation: Check if `phone1` or `id_proof_number` already exists in the database
    const existingPatient = await PR_patientReg.findOne({
      where: {
        [Op.or]: [
          { phone1: pdata.phone1 },
          { id_proof_number: pdata.id_proof_number },
        ],
      },
    });

    if (existingPatient) {
      return res.status(400).json({
        msg: "A patient with the same phone number or ID proof number already exists.",
      });
    }

    // If files are uploaded, add file paths to patient data
    if (file) {
      if (file.patientImg)
        pdata = { ...pdata, image_path: file.patientImg[0].filename };
      if (file.spouseImg)
        pdata = { ...pdata, spouse_path: file.spouseImg[0].filename };
    }

    // Proceed to create the new patient record in the database
    const data = await PR_patientReg.create(pdata);

    // After the patient is created, generate 'mr_no' by prefixing 'MR_' and the created patient's 'id'
    const mr_no = `MR_${data.id}`;

    // Update the created patient record with the generated 'mr_no'
    await data.update({ mr_no: mr_no });

    // Respond with success message if the data is saved successfully
    return res.status(200).json({
      msg: "Patient registration completed successfully.",
      mr_no: mr_no, // Return the generated mr_no in the response
    });
  } catch (error) {
    console.error("Error in patient registration:", error);
    return res
      .status(500)
      .json({ msg: "Internal Server Error.", error: error.message });
  }
};

const updatePatientRegistrationDetail = async (req, res) => {
  try {
    const { patientId: id } = req.session;
    const { patientData } = req.body;

    // Check if no patient data is provided
    if (!id) {
      return res
        .status(404)
        .send({ message: "Patient not found with given patient ID." });
    }
    // Find patient with the given ID
    const patient = await PR_patientReg.findOne({ where: { id } });

    // Check if patient is not found
    if (!patient) {
      return res
        .status(404)
        .send({ message: "Patient not found with given Patient ID." });
    }

    // Update patient data
    await patient.update(patientData);
    await patient.save();

    res.status(200).send();
  } catch (error) {
    console.log("Error while updating patient registration");
    return res.status(500).send({ message: error.message });
  }
};

const AppointmentData = async (req, res) => {
  try {
    // Extract MR number from query parameters
    const mrNo = req.query.mrNo;
    console.log("Received MR No:", mrNo); // More descriptive console log

    // Find patient based on MR number
    const patient = await PR_patientReg.findOne({
      where: { mr_no: mrNo },
    });

    // Check if the patient was found
    if (!patient) {
      console.log(`No patient found with MR No: ${mrNo}`); // Logging for debugging
      return res
        .status(404)
        .json({ msg: "No patient found with this MR number." }); // 404 is more specific for not found
    }

    // If patient is found, perhaps do something with the patient data
    // For example, send the patient data back or proceed with additional processing
    res.json(patient); // Assuming you want to send the patient data back
  } catch (error) {
    console.error("Error fetching patient data:", error.message); // Enhanced error logging
    return res
      .status(500)
      .json({ msg: "Internal Server Error. Please try again later." });
  }
};

const newAppointment = async (req, res) => {
  try {
    const appointment = req.body;
    console.log(appointment);

    const patient = await PR_patientReg.findOne({
      where: { mr_no: appointment.mrNo },
    });
    console.log(patient);

    if (!patient) {
      return res
        .status(400)
        .json({ msg: "No Patient with this MR registration..." });
    }

    const data = await PR_Appointment.create({
      ...appointment,
    });
    console.log(data);

    return res.status(200).json({ msg: "Appointment created successfully..." });
  } catch (error) {
    console.log(error.message);
    return res.status(500).json({ msg: "Internal Server Error..." });
  }
};

const newVisit = async (req, res) => {
  try {
    const visit = req.body;
    console.log(visit);

    const data = await PR_PatientVisit.create({
      ...visit,
    });
    console.log(data);

    // Store mrNo in the session
    req.session.mrNo = data.mrNo;
    console.log(`mrNo ${data.mrNo} stored in session`);

    return res.status(200).json({ msg: "Visit created successfully..." });
  } catch (error) {
    console.log(error.message);
    return res.status(500).json({ msg: "Internal Server Error..." });
  }
};

const newCouple = async (req, res) => {
  try {
    const { mrNo } = req.body;

    // Check if mrNo already exists
    const existingRecord = await PR_formNewCouple.findOne({
      where: { mrNo: mrNo },
    });

    if (existingRecord) {
      // If mrNo exists, return a conflict response

      return res.status(409).send("MR No. already exists");
    }

    // If mrNo does not exist, create a new entry
    await PR_formNewCouple.create(req.body);

    res.status(200).send("Data saved successfully");
  } catch (error) {
    console.error("Failed to save data:", error);
    res.status(500).send("Internal Server Error");
  }
};

const billPatientSubmit = async (req, res) => {
  const clinicId = req.user.clinicId; // Get clinic_id from session

  if (clinicId === null) {
    return res
      .status(400)
      .send({ msg: "Clinic ID is missing in the session." });
  }

  req.body.clinic_id = clinicId; // Use the clinic_id from session
  console.log("Request Body:", req.body);
  const transaction = await sequelize.transaction(); // Start a transaction
  try {
    const {
      query,
      doctor,
      freeze,
      totalConcessionAmount,
      selectedBillServices,
      paymentDetails,
      ...updateData
    } = req.body;

    const approvalCondition =
      freeze === true && parseFloat(totalConcessionAmount) === 0;

    const receivedAmount = parseFloat(paymentDetails.recievedAmount) || 0;
    const adjustAdvancePayment =
      parseFloat(paymentDetails.adjustAdvancePayment) || 0;
    let balanceAmount;

    if (query == 1) {
      // Updating an existing bill
      const { itemid } = req.body;

      const bill = await PR_BillFindPatient.findOne({ where: { id: itemid } });
      if (!bill) {
        throw new Error("Patient record not found");
      }

      const previousBalance = parseFloat(bill.balanceAmount);
      balanceAmount = previousBalance - receivedAmount;

      await PR_BillFindPatient.update(
        {
          ...updateData,
          freeze,
          totalConcessionAmount,
          selectedBillServices,
          balanceAmount: balanceAmount.toFixed(2),
          approved: approvalCondition ? true : req.body.approved,
        },
        { where: { id: itemid }, transaction }
      );

      // await handleBillServices(itemid, selectedBillServices, approvalCondition, transaction);

      await PR_BillPaymentDetails.create(
        {
          bill_no: `BILL${itemid}`,
          ...paymentDetails,
        },
        { transaction }
      );

      await deductFromAdvance(
        updateData.patient_id,
        adjustAdvancePayment,
        transaction
      );
    } else {
      // Creating a new bill
      balanceAmount =
        parseFloat(updateData.totalNetBillAmount) - receivedAmount;

      const newBill = await PR_BillFindPatient.create(
        {
          ...updateData,
          freeze,
          doctor,
          date: new Date(),
          totalConcessionAmount,
          selectedBillServices,
          balanceAmount: balanceAmount.toFixed(2),
          approved: approvalCondition ? true : false,
        },
        { transaction }
      );

      const billNo = `BILL${newBill.id}`;
      await newBill.update({ bill_no: billNo }, { transaction });

      // await handleBillServices(newBill.id, selectedBillServices, approvalCondition, transaction);

      await PR_BillPaymentDetails.create(
        {
          bill_no: billNo,
          ...paymentDetails,
        },
        { transaction }
      );

      await deductFromAdvance(
        updateData.patient_id,
        adjustAdvancePayment,
        transaction
      );
    }

    await transaction.commit(); // Commit the transaction
    res.status(200).send("Transaction completed successfully.");
  } catch (error) {
    console.error("Transaction failed:", error);
    await transaction.rollback(); // Rollback on error
    res.status(500).send({
      message: "An error occurred while processing the transaction.",
      details: error.message,
    });
  }
};

const setSessionPatientId = async (req, res) => {
  try {
    const { id } = req.query;
    req.session.patientId = id;
    req.session.rowId = id;
    const patientRecord = await PatientCounseling.findOne({
      where: {
        patientId: req.session.patientId,
        status: true,
        clinicId: req.user.clinicId,
      },
      attributes: ["cycle_id", "mrn"],
    });
    if (patientRecord.mrn) {
      req.session.cycle_id = patientRecord.cycle_id;
    }
    console.log("Setted patient id: ", req.session.patientId);

    res.status(200).json({ msg: "PatientId set successfully" });
  } catch (err) {
    console.log(err);
    return res
      .status(400)
      .json({ msg: "Error in extracting data from query parameters." });
  }
};

const billPharmacyPatientSubmit = async (req, res) => {
  const clinicId = req.user.clinicId; // Get clinic_id from session

  if (clinicId === null) {
    return res
      .status(400)
      .send({ msg: "Clinic ID is missing in the session." });
  }

  req.body.clinic_id = clinicId; // Use the clinic_id from session
  console.log(req.body);
  const transaction = await sequelize.transaction(); // Start a transaction
  try {
    const {
      query,
      freeze,
      totalConcessionAmount,
      selectedBillItems,
      paymentDetails,
      ...updateData
    } = req.body;

    const approvalCondition =
      freeze === true && parseFloat(totalConcessionAmount) === 0;

    const receivedAmount = parseFloat(paymentDetails.recievedAmount) || 0;
    const adjustAdvancePayment =
      parseFloat(paymentDetails.adjustAdvancePayment) || 0;
    let balanceAmount;

    if (query == 1) {
      const { itemid } = req.body;

      const bill = await PR_BillPharmacy.findOne({ where: { id: itemid } });
      if (!bill) {
        throw new Error("Patient record not found");
      }

      const previousBalance = parseFloat(bill.balanceAmount);
      balanceAmount = previousBalance - receivedAmount;

      // Update existing bill
      await PR_BillPharmacy.update(
        {
          ...updateData,
          freeze,
          totalConcessionAmount,
          selectedBillItems,
          balanceAmount: balanceAmount.toFixed(2),
          approved: approvalCondition ? true : req.body.approved,
        },
        { where: { id: itemid }, transaction }
      );

      // Save payment details
      await PR_BillPaymentDetails.create(
        {
          bill_no: `BILLPH${itemid}`,
          ...paymentDetails,
        },
        { transaction }
      );

      // Deduct advance payments
      await deductFromAdvance(
        updateData.patient_id,
        adjustAdvancePayment,
        transaction
      );

      // Update stock quantities
      // await updateStockQuantities(selectedBillItems, transaction, updateData.store);

      await transaction.commit(); // Commit the transaction
      res.status(200).send("Data updated successfully, payment details saved.");
    } else {
      // Create a new bill
      balanceAmount =
        parseFloat(updateData.totalNetBillAmount) - receivedAmount;

      const newBill = await PR_BillPharmacy.create(
        {
          ...updateData,
          freeze,
          date: new Date(),
          totalConcessionAmount,
          selectedBillItems,
          balanceAmount: balanceAmount.toFixed(2),
          approved: approvalCondition ? true : false,
        },
        { transaction }
      );

      // Generate and save bill number
      const billNo = `BILLPH${newBill.id}`;
      await newBill.update({ bill_no: billNo }, { transaction });

      // Save payment details
      await PR_BillPaymentDetails.create(
        {
          bill_no: billNo,
          ...paymentDetails,
        },
        { transaction }
      );

      // Deduct advance payments
      await deductFromAdvance(
        updateData.patient_id,
        adjustAdvancePayment,
        transaction
      );

      // Update stock quantities
      await updateStockQuantities(
        selectedBillItems,
        transaction,
        updateData.store
      );

      await transaction.commit(); // Commit the transaction
      res.status(200).send("Data saved successfully with Bill No: " + billNo);
    }
  } catch (error) {
    console.error("Failed to save data:", error);
    await transaction.rollback(); // Rollback transaction in case of errors

    // Send error details to the user
    res.status(400).send({
      msg: error.message || "An error occurred while processing the request.",
      details: error.details || null,
    });
  }
};

/**
 * Deducts the given amount from the patient's advance payments starting from the first entry.
 * If the first entry doesn't have enough balance, deduct from subsequent entries.
 *
 * @param {string} patientId - The patient ID for which advance payments need to be adjusted.
 * @param {number} amountToDeduct - The total amount to deduct from advances.
 * @param {object} transaction - The Sequelize transaction object.
 */
const deductFromAdvance = async (patientId, amountToDeduct, transaction) => {
  const advanceRecords = await Advance.findAll({
    where: { patient_id: patientId },
    order: [["createdAt", "ASC"]], // Deduct from the oldest entries first
    transaction,
  });

  for (const record of advanceRecords) {
    const recordBalance = parseFloat(record.balance_amount);
    if (amountToDeduct <= 0) break; // Stop if the total amount has been deducted

    if (recordBalance >= amountToDeduct) {
      // Current record has enough balance to cover the amount
      const updatedBalance = recordBalance - amountToDeduct;
      await record.update(
        {
          consume_amount: parseFloat(record.consume_amount) + amountToDeduct,
          balance_amount: updatedBalance.toFixed(2),
        },
        { transaction }
      );
      amountToDeduct = 0; // Deduction is complete
    } else {
      // Deduct the remaining balance from the current record
      await record.update(
        {
          consume_amount: parseFloat(record.consume_amount) + recordBalance,
          balance_amount: 0,
        },
        { transaction }
      );
      amountToDeduct -= recordBalance; // Deduct the remaining amount from the next record
    }
  }

  if (amountToDeduct > 0) {
    throw new Error(
      "Insufficient advance balance to complete the transaction."
    );
  }
};

async function updateStockQuantities(
  selectedBillItems,
  transaction,
  storeName
) {
  for (const item of selectedBillItems) {
    const { item_code, batch_code, quantity } = item;

    try {
      // Log item details for debugging
      console.log(
        `Updating stock for item_code: ${item_code}, batch_code: ${batch_code}, quantity: ${quantity}, store_name: ${storeName}`
      );

      // Find the current stock for the item, batch, and store
      const stock = await CurrentItemStock.findOne({
        where: {
          item_code,
          batch_code,
          store_name: storeName, // Include store_name in the query
        },
      });

      // Log the fetched stock for debugging
      console.log(stock);

      if (!stock) {
        throw new Error(
          `Stock not found for item_code "${item_code}", batch "${batch_code}", and store "${storeName}".`
        );
      }

      // Log the available stock for debugging
      console.log(
        `Available stock for item_code "${item_code}", batch "${batch_code}" in store "${storeName}": ${stock.available_stock}`
      );

      if (stock.available_stock < quantity) {
        throw new Error(
          `Insufficient stock: Requested ${quantity}, Available ${stock.available_stock} for item_code "${item_code}" in batch "${batch_code}" and store "${storeName}".`
        );
      }

      // Subtract the issued quantity from the available stock
      await stock.update(
        { available_stock: stock.available_stock - quantity },
        { transaction }
      );

      // Log the new stock level after update
      console.log(
        `Updated stock for item_code "${item_code}", batch "${batch_code}" in store "${storeName}": Remaining stock ${
          stock.available_stock - quantity
        }`
      );
    } catch (error) {
      // Include details in the error to propagate to the main function
      error.details = { item_code, batch_code, quantity, storeName };
      throw error; // Propagate the error back to the main function
    }
  }
}

const handleBillItems = async (billId, selectedBillItems) => {
  try {
    // Remove any existing items for this bill ID before inserting new ones
    await BillItems.destroy({
      where: { billId },
    });

    // Add the new selected items
    const itemsToInsert = selectedBillItems.map((item) => ({
      billId: billId, // Associate the items with the bill ID
      item_code: item.item_code,
      item_name: item.item_name,
      batch_code: item.batch_code,
      hsn_code: item.hsn_code,
      expiry_date: item.expiry_date,
      rate: item.rate,
      quantity: item.quantity,
      concession_per: item.concession_per,
      concession_amount: item.concession_amount,
      totalAmount: item.totalAmount,
      netAmount: item.netAmount,
      doctor: item.doctor || null,
      gst: item.gst,
      // Associate a doctor if applicable
    }));

    // Bulk insert the new items
    await BillItems.bulkCreate(itemsToInsert);
  } catch (error) {
    console.error("Failed to handle bill items:", error);
    throw new Error("Error saving items");
  }
};

// Helper function to save or update the selectedBillServices in BillServices
const handleBillServices = async (billId, selectedBillServices) => {
  try {
    // Remove any existing services for this bill ID before inserting new ones
    await BillServices.destroy({
      where: { billId },
    });

    // Add the new selected services
    const servicesToInsert = selectedBillServices.map((service) => ({
      billId, // Associate the services with the bill ID
      serviceCode: service.serviceCode,
      serviceName: service.serviceName,
      serviceRate: service.serviceRate,
      concession_per: service.concession_per,
      concession_amount: service.concession_amount,
      service_amount: service.service_amount,
      totalAmount: service.totalAmount,
      netAmount: service.netAmount,
      doctor: service.doctor || null,
    }));

    // Bulk insert the new services
    await BillServices.bulkCreate(servicesToInsert);
  } catch (error) {
    console.error("Failed to handle bill services:", error);
    throw new Error("Error saving services");
  }
};

const SaveStatusData = async (req, res) => {
  try {
    console.log(req.body);
    const {
      id: encryptedId,
      status,
      schema: encryptedSchema,
      tableStatus: EnctableStatus,
    } = req.body;
    const secretKey = "ll"; // Replace with your actual secret key

    // Decrypt id and schema
    const id = decryptData(decodeURIComponent(encryptedId), secretKey); // URL decode
    const schema = decryptData(decodeURIComponent(encryptedSchema), secretKey); // URL decode
    const tableStatus = decryptData(
      decodeURIComponent(EnctableStatus),
      secretKey
    ); // URL decode

    console.log("Decrypted id:", id);
    console.log("Decrypted schema:", schema);
    console.log("Decrypted tableStatus:", tableStatus);

    // Use decrypted id and schema to fetch data from the model
    const Model = require("../models/PatientReg")[schema];
    const data = await Model.findByPk(id);

    if (data) {
      ``;

      // Update status and save
      data[tableStatus] = status;

      await data.save();
      console.log(data);
      res.sendStatus(200);
    } else {
      res.status(404).send("Data not found");
    }
  } catch (error) {
    console.error("Error updating status:", error);
    res.status(500).send("Internal Server Error");
  }
};

const getpatientBills = async (req, res) => {
  try {
    if (!req.session.rowId) {
      return res.status(400).json({ error: "Session rowId is not available." });
    }

    // Fetch the patient using the rowId from the session
    const patient = await PR_patientReg.findByPk(req.session.rowId);
    if (!patient) {
      return res.status(404).json({ error: "Patient not found." });
    }

    const patientMrNo = patient.mr_no;

    /* Fetch records matching the patient's MR number from PR_BillFindPatient */
    const encModData = await PR_BillFindPatient.findAll({
      where: {
        mrNo: patientMrNo,
      },
    });

    // Encrypt the id and prepare the response
    const details = encModData.map((data) => {
      const encryptedId = encryptDataForUrl(data.id.toString());
      return {
        ...data.toJSON(),
        id: encryptedId,
      };
    });

    res.status(200).json(details);
  } catch (error) {
    console.error("Error fetching patient bills:", error);
    res
      .status(500)
      .json({ error: "An error occurred while fetching patient bills." });
  }
};

const {
  BankMaster,
  BankBranchMaster,
  Department,
  Employee,
} = require("../models/clinicConfig");

const { PrefixMaster } = require("../models/patientConfig");

const { Doctor, bcTariffMasterNew } = require("../models/billingSchema");
const { CurrentItemStock } = require("../models/mainInvSchema");
const { PatientCounseling } = require("../models/embrology");

const getPrefixMaster = async (req, res) => {
  try {
    const prefixDetails = await PrefixMaster.findAll();
    console.log(prefixDetails);
    res.status(200).json(prefixDetails);
  } catch (e) {
    res.status(500).json({
      success: false,
      message: e,
    });
  }
};

const getBankNames = async (req, res) => {
  try {
    const bankNames = await BankMaster.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["bank_m_desc"],
    });
    const branchNames = await BankBranchMaster.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["bank_branch_m_desc"],
    });
    const departmentNames = await Department.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["dept_desc"],
    });
    const doctorNames = await Doctor.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["doc_name"],
    });
    const tarriffNames = await bcTariffMasterNew.findAll({
      where: {
        clinic_id: req.user.clinicId,
      },
      attributes: ["trf_name"],
    });
    const employeeNames = await Employee.findAll({
      where: {
        clinic_id: req.user.clinicId,
        emp_desig: "Agent",
      },
      attributes: ["emp_first_name"],
    });
    res.status(200).json({
      bankNames,
      branchNames,
      departmentNames,
      doctorNames,
      tarriffNames,
      employeeNames,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: error.message,
    });
  }
};

const search = async (req, res) => {
  try {
    const { searchTerm } = req.query;

    const patients = await PR_patientReg.findAll({
      where: {
        clinic_id: req.user.clinicId,
        [Op.or]: [
          {
            firstName: {
              [Op.like]: `${searchTerm}%`, // Searches by first name, case-insensitive for MySQL
            },
          },
          {
            mr_no: {
              [Op.like]: `${searchTerm}%`, // Searches by MR number, case-insensitive for MySQL
            },
          },
          {
            phone1: {
              [Op.like]: `${searchTerm}%`, // Searches by phone, case-insensitive for MySQL
            },
          },
        ],
      },
      attributes: ["id", "firstName", "lastName", "phone1", "Gender"],
    });

    if (patients.length > 0) {
      return res.status(200).json(patients);
    }

    const inventory = await ItemMasterNew.findAll({
      where: {
        [Op.or]: [
          {
            molecule_name: {
              [Op.like]: `%${searchTerm}%`, // Partial match with case-insensitive search for MySQL
            },
          },
          {
            item_name: {
              [Op.like]: `%${searchTerm}%`, // Partial match with case-insensitive search for MySQL
            },
          },
        ],
        clinic_id: req.user.clinicId,
      },
    });

    if (inventory.length > 0) {
      return res.status(200).json(inventory);
    }

    const bills = await PR_BillFindPatient.findAll({
      where: {
        bill_no: {
          [Op.like]: `%${searchTerm}%`,
        },
      },
    });
    if (bills.length > 0) {
      return res.status(200).json(bills);
    }

    return res.status(404).json({ message: "No results found." });
  } catch (err) {
    res.status(500).json({ error: "An error occurred while searching" });
    console.log(err);
  }
};

module.exports = {
  patientRagistration,
  updatePatientRegistrationDetail,
  AppointmentData,
  newAppointment,
  newVisit,
  newCouple,
  billPatientSubmit,
  SaveStatusData,
  getpatientBills,
  billPharmacyPatientSubmit,
  getBankNames,
  getPrefixMaster,
  search,
  setSessionPatientId,
};
