const { DataTypes } = require("sequelize");
const { sequelize, con } = require("../sequelize");
const flash = require("express-flash");
const { PR_patientReg } = require("./PatientReg");
const { response } = require("express");

const ServiceMaster = sequelize.define(
  "ServiceMaster",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    ser_code: {
      type: DataTypes.STRING,
    },
    ser_code_type: {
      type: DataTypes.STRING,
    },
    ser_code_details: {
      type: DataTypes.STRING,
    },
    ser_spec: {
      type: DataTypes.STRING,
    },
    ser_sub_spec: {
      type: DataTypes.STRING,
    },
    ser_sac_code: {
      type: DataTypes.STRING,
    },
    ser_short_desc: {
      type: DataTypes.STRING,
    },
    ser_long_desc: {
      type: DataTypes.STRING,
    },
    ser_name: {
      type: DataTypes.STRING,
    },
    ser_base_rate: {
      type: DataTypes.DECIMAL,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "service_masters", // Specify the table name
  }
);

const BCExpMaster = sequelize.define(
  "BCExpMaster",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    userId: {
      type: DataTypes.INTEGER,
    },
    exp_master_code: {
      type: DataTypes.STRING,
    },
    exp_master_desc: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "exp_masters", // Specify the table name
  }
);

const getProvisionalBills = async (mrNo) => {
  try {
    console.log(`Fetching provisional bills for MR No: ${mrNo}`);

    // Query for Pharmacy Bills
    const pharmacyResults = await sequelize.query(
      'SELECT * FROM pr_billpharmacy WHERE mrNo = :mrNo AND freeze="1" AND approved="0"',
      { replacements: { mrNo }, type: sequelize.QueryTypes.SELECT }
    );
    const mappedPharmacyBills = pharmacyResults.map((bill) => ({
      billId: bill.bill_no,
      date: bill.date,
      totalAmount: bill.totalNetBillAmount,
      balanceAmount: bill.balanceAmount,
      type: 'Provisional',
    }));

    // Query for Service Bills
    const serviceResults = await sequelize.query(
      'SELECT * FROM pr_billfindpatient WHERE mrNo = :mrNo AND freeze="1" AND approved="0"',
      { replacements: { mrNo }, type: sequelize.QueryTypes.SELECT }
    );
    const mappedServiceBills = serviceResults.map((bill) => ({
      billId: bill.bill_no,
      date: bill.date,
      totalAmount: bill.totalNetAmount,
      balanceAmount: bill.balanceAmount,
      type: 'Provisional',
    }));

    // Combine Pharmacy and Service Bills
    const provisionalBills = [...mappedPharmacyBills, ...mappedServiceBills];
    console.log('Provisional Bills:', provisionalBills);

    return provisionalBills;
  } catch (error) {
    console.error('Error fetching provisional bills:', error.message, error.stack);
    throw error;
  }
};
const getServiceBills = async (mrNo) => {
  try {
    const results = await sequelize.query(
      'SELECT * FROM pr_billfindpatient WHERE mrNo = :mrNo and freeze="1" and approved="1"',
      { replacements: { mrNo }, type: sequelize.QueryTypes.SELECT }
    );
    // Map attributes to match table headings
    const mappedResults = results.map((bill) => ({
      billId: bill.bill_no,
      date: bill.date,
      totalAmount: bill.totalNetAmount,
      balanceAmount: bill.balanceAmount,
      type: 'Service',
    }));
    //console.log('Mapped Service Bills:', mappedResults);
    return mappedResults;
  } catch (error) {
    console.error('Error fetching service bills:', error.message, error.stack);
    throw error;
  }
};

const getPharmacyBills = async (mrNo) => {
  try {
    const results = await sequelize.query(
      'SELECT * FROM pr_billpharmacy WHERE mrNo = :mrNo and freeze="1" and approved="1"',
      { replacements: { mrNo }, type: sequelize.QueryTypes.SELECT }
    );
    // Map attributes to match table headings
    const mappedResults = results.map((bill) => ({
      billId: bill.bill_no,
      date: bill.date,
      totalAmount: bill.totalNetBillAmount,
      balanceAmount: bill.balanceAmount,
      type: 'Pharmacy',
    }));
    //console.log('Mapped Pharmacy Bills:', mappedResults);
    return mappedResults;
  } catch (error) {
    console.error('Error fetching pharmacy bills:', error.message, error.stack);
    throw error;
  }
};

const getPackageBills = async (mrNo) => {
  try {
    const patientId = mrNo.replace(/\D+/g, ""); 

    console.log("Fetching package bills for patientId:", patientId);
    const results = await sequelize.query(
      `SELECT 
        bill_no, 
        DATE_FORMAT(date, '%Y-%m-%d') AS date, 
        total_package_amount, 
        balance_amount AS balance
      FROM bifurcation_stages 
      WHERE patientId = :patientId`,
      {
        replacements: { patientId },
        type: sequelize.QueryTypes.SELECT
      }
    );

    const mappedResults = results.map((bill) => ({
      billId: bill.bill_no,
      date: bill.date,
      totalAmount: (bill.total_package_amount),
      balanceAmount: bill.balance,
      type: 'Package',
    }));

    console.log('Mapped Package Bills:', mappedResults);
    return mappedResults;
  } catch (error) {
    console.error('Error fetching package bills:', error.message, error.stack);
    throw error;
  }
};

const getCombinedBills = async (mrNo) => {
  try {
    // console.log("--------- Hiiiii", mrNo);
    const patientId = mrNo.replace('MR_', '')
    console.log(patientId);

    console.log(`Fetching bills for MR No: ${mrNo}`);
    const serviceBills = await getServiceBills(mrNo);
    console.log('Service Bills:', serviceBills);
    const pharmacyBills = await getPharmacyBills(mrNo);
    console.log('Pharmacy Bills:', pharmacyBills);
    const provisionalBills = await getProvisionalBills(mrNo);
    console.log('Provisional Bills:', provisionalBills);
    const packageBills = await getPackageBills(patientId);
    console.log('Package Bills:', packageBills);
    const combinedBills = [...serviceBills, ...pharmacyBills, ...provisionalBills, ...packageBills];
    console.log('Combined Bills:', combinedBills);
    return { serviceBills, pharmacyBills, provisionalBills, packageBills, combinedBills };
  } catch (error) {
    console.error('Error fetching combined bills:', error.message, error.stack);
    throw error;
  }
};

const getPatientIdFromMRNo = async (mrNo) => {
  try {
    const result = await sequelize.query(
      'SELECT id FROM pr_patientreg WHERE mr_no = :mrNo',
      { replacements: { mrNo }, type: sequelize.QueryTypes.SELECT }
    );

    if (result.length === 0) {
      throw new Error(`No patient found for MR No: ${mrNo}`);
    }

    return result[0].id;
  } catch (error) {
    console.error('Error fetching patient_id:', error.message);
    throw error;
  }
};

const getTransactionReceipts = async (mrNo, receiptType = 'all', paymentType = 'all') => {
  try {
    console.log(`Fetching receipts for: ${mrNo}`);
    const patientId = await getPatientIdFromMRNo(mrNo);
    console.log(`Patient ID: ${patientId}`);

    const queries = [
      { table: 'mainpatientadvance', receiptType: 'Advance' },
      { table: 'mainpatientrefund', receiptType: 'Refund' },
      { table: 'maincompany_advance', receiptType: 'Advance' },
      { table: 'maincompanyrefunds', receiptType: 'Refund' },
      // { table: 'pr_billpharmacy', receiptType: 'Advance' },
      // { table: 'pr_billfindpatient', receiptType: 'Advance' },
    ];
    const transactionReceipts = [];

    for (const query of queries) {
      const results = await sequelize.query(
        `SELECT * FROM ${query.table} WHERE patient_id = :patientId`,
        { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
      );
      // const results = await sequelize.query(
      //   `SELECT * FROM ${query.table} WHERE ${query.filterBy} = :value`,
      //   { replacements: { value: query.value }, type: sequelize.QueryTypes.SELECT }
      // );
      console.log(`Results for table ${query.table}:`, results);
      console.log('Before Loop:', transactionReceipts);
    }
    for (const query of queries) {
      console.log(`Querying ${query.table} for patient ID: ${patientId}`);
      const results = await sequelize.query(
        `SELECT * FROM ${query.table} WHERE patient_id = :patientId`,
        { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
      );

      console.log(`Results for ${query.table}:`, results);

      const mappedResults = results.map((receipt) => ({
        rec_no: receipt.rec_no,
        updatedAt: receipt.updatedAt,
        receiptType: query.receiptType,
        paymentMethod: receipt.paymentMethod || 'NEFT/RTGS', // Default if missing
        totalAmount: receipt.amount || receipt.totalAdvance || receipt.totalNetAmount,
        balanceAmount: receipt.balance_amount || receipt.balanceAmount,
      }));

      transactionReceipts.push(...mappedResults);
    }

    console.log('Combined Transaction Receipts:', transactionReceipts);

    // Filter based on receiptType and paymentType
    return transactionReceipts.filter((receipt) => {
      const matchesReceiptType = receiptType === 'all' || receipt.receiptType === receiptType;
      const matchesPaymentType = paymentType === 'all' || receipt.paymentMethod === paymentType;
      return matchesReceiptType && matchesPaymentType;
    });
  } catch (error) {
    console.error('Error fetching transaction receipts:', error.message);
    throw error;
  }
};

const getReceiptDetails = async (receiptNo) => {
  try {
    const tables = [
      { table: 'mainpatientadvance', receiptType: 'Advance' },
      { table: 'mainpatientrefund', receiptType: 'Refund' },
      { table: 'maincompany_advance', receiptType: 'Advance' },
      { table: 'maincompanyrefunds', receiptType: 'Refund' },
    ];
    let receipt = null;
    for (const { table, receiptType } of tables) {
      const query = `SELECT * FROM ${table} WHERE rec_no = :receiptNo`;
      const results = await sequelize.query(query, {
        replacements: { receiptNo },
        type: sequelize.QueryTypes.SELECT,
      });
      console.log(`Results for ${query.table}:`, results);
      if (results.length > 0) {
        receipt = results[0];
        receipt.receiptType = receiptType;
        break;
      }
    }
    const patientId = receipt.patient_id;
    let patientDetails = null;

    if (patientId) {
      const patient = await PR_patientReg.findOne({
        where: { id: patientId },
        attributes: ['firstName', 'lastName', 'mr_no'],
      });
      if (patient) {
        patientDetails = {
          patientName: `${patient.firstName} ${patient.lastName}`,
          mr_no: patient.mr_no,
        };
      } else {
        console.error(`No patient found with ID: ${patientId}`);
      }
    }
    if (!receipt) {
      throw new Error('Receipt not found');
    }
    return {
      ...receipt,
      patientDetails,
    };
  } catch (error) {
    console.error('Error fetching receipt details:', error);
    throw error;
  }
};

// const getSapData = async (mrNo) => {
//   try {
//     const patientId = await getPatientIdFromMRNo(mrNo);

//     // Fetch Service Advices and Prescriptions
//     const serviceAdvices = await sequelize.query(
//       'SELECT Date, ServiceAdvice FROM emb_serviceadvices WHERE patientId = :patientId',
//       { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
//     );

    


//     const serviceDescription = serviceAdvices[0]?.ServiceAdvice?.[0]?.service || 'No Description';

//     console.log(serviceDescription);
//     const prescriptions = await sequelize.query(
//       'SELECT Date, prescribedMedicine FROM emb_prescription WHERE patientId = :patientId',
//       { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
//     );

//     const prescriptionDescriptions = prescriptions.map(prescription => {
//       const prescribedMedicine = prescription.prescribedMedicine;
//       if (prescribedMedicine && Array.isArray(prescribedMedicine) && prescribedMedicine.length > 0) {
//         return prescribedMedicine[0];
//       } else {
//         return {};
//       }
//     });
    
//     const combinedData = [
//       ...serviceAdvices.map(item => ({
//         Date: item.Date,
//         type: 'Service Advice',
//         description: item.ServiceAdvice?.[0]?.service || 'No Description' // Extract service
//       })),
//       ...prescriptions.map((item, index) => ({
//         Date: item.Date,
//         type: 'Prescription',
//         description: prescriptionDescriptions[index] || 'No Item Name'
//       }))
//     ];
        
//     return combinedData;
//   } catch (error) {
//     console.error('Error fetching SAP data:', error.message);
//     throw error;
//   }
// };

// const getSapData = async (mrNo) => {
//   try {
//     const patientId = await getPatientIdFromMRNo(mrNo);

//     // Fetch Service Advices and Prescriptions
//     const serviceAdvices = await sequelize.query(
//       'SELECT Date, ServiceAdvice, createdAt FROM emb_serviceadvices WHERE patientId = :patientId',
//       { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
//     );

//     const prescriptions = await sequelize.query(
//       'SELECT Date, prescribedMedicine, createdAt FROM emb_prescription WHERE patientId = :patientId',
//       { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
//     );

//     // Process Service Advice
//     const serviceAdviceData = {};
//     serviceAdvices.forEach(item => {
//       const date = item.Date;
//       if (!serviceAdviceData[date]) {
//         serviceAdviceData[date] = [];
//       }
//       serviceAdviceData[date].push(
//         ...(Array.isArray(item.ServiceAdvice) ? item.ServiceAdvice.map(sa => sa.service) : ['No Description'])
//       );
//     });

//     // Process Prescriptions
//     const prescriptionData = {};
//     prescriptions.forEach(item => {
//       const date = item.Date;
//       if (!prescriptionData[date]) {
//         prescriptionData[date] = [];
//       }
//       if (Array.isArray(item.prescribedMedicine) && item.prescribedMedicine.length > 0) {
//         prescriptionData[date].push(...item.prescribedMedicine);
//       } else {
//         prescriptionData[date].push({});
//       }
//     });

//     return {
//       serviceAdvices: Object.keys(serviceAdviceData).map(date => ({
//         Date: date,
//         advices: serviceAdviceData[date] || []
//       })),
//       prescriptions: Object.keys(prescriptionData).map(date => ({
//         Date: date,
//         medicines: prescriptionData[date] || []
//       }))
//     };
//   } catch (error) {
//     console.error('Error fetching SAP data:', error.message);
//     throw error;
//   }
// };


const getSapData = async (mrNo) => {
  try {
    const patientId = await getPatientIdFromMRNo(mrNo);

    // Fetch Service Advices and Prescriptions
    const serviceAdvices = await sequelize.query(
      'SELECT Date, ServiceAdvice, createdAt FROM emb_serviceadvices WHERE patientId = :patientId',
      { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
    );

    const prescriptions = await sequelize.query(
      'SELECT Date, prescribedMedicine, createdAt FROM emb_prescription WHERE patientId = :patientId',
      { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
    );

    // Process Service Advice
    const serviceAdviceData = {};
    serviceAdvices.forEach(item => {
      const date = item.Date;
      if (!serviceAdviceData[date]) {
        serviceAdviceData[date] = [];
      }
      serviceAdviceData[date].push({
        services: Array.isArray(item.ServiceAdvice) ? item.ServiceAdvice.map(sa => sa.service) : ['No Description'],
        createdAt: new Date(item.createdAt)
      });
    });

    // Process Prescriptions
    const prescriptionData = {};
    prescriptions.forEach(item => {
      const date = item.Date;
      if (!prescriptionData[date]) {
        prescriptionData[date] = [];
      }
      prescriptionData[date].push({
        medicines: Array.isArray(item.prescribedMedicine) && item.prescribedMedicine.length > 0
          ? item.prescribedMedicine
          : [{}],
        createdAt: new Date(item.createdAt)
      });
    });

    // Merge and Sort Data by `createdAt`
    const mergedData = [
      ...Object.keys(serviceAdviceData).flatMap(date => serviceAdviceData[date].map(sa => ({
        Date: date,
        type: 'Service Advice',
        advices: sa.services,
        createdAt: sa.createdAt
      }))),
      ...Object.keys(prescriptionData).flatMap(date => prescriptionData[date].map(pres => ({
        Date: date,
        type: 'Prescription',
        medicines: pres.medicines,
        createdAt: pres.createdAt
      })))
    ];

    // Sort by createdAt (latest first)
    mergedData.sort((a, b) => b.createdAt - a.createdAt);

    return mergedData;
  } catch (error) {
    console.error('Error fetching SAP data:', error.message);
    throw error;
  }
};



const getCycleIdFromPatientId = async (mrNo) => {
  try {
    const patientId = await getPatientIdFromMRNo(mrNo);
    console.log(`Patient ID for MR No ${mrNo}:`, patientId);
    const result = await sequelize.query(
      'SELECT cycle_id FROM patient_counseling WHERE patientId = :patientId',
      { replacements: { patientId }, type: sequelize.QueryTypes.SELECT }
    );
    console.log('Result from querying patient_counseling:', result);
    if (result.length === 0) {
      throw new Error(`No cycle ID found for patient with MR No: ${mrNo}`);
    }
    // console.log(`Cycle ID for patient ${mrNo}:`, result[0].cycle_id);
    // return result[0].cycle_id;
    const cycleData = result.map((cycle, index) => ({
      cycle_id: cycle.cycle_id,
      cycle_name: `Cycle ${index + 1}`
    }));

    return cycleData;
  } catch (error) {
    console.error('Error fetching cycle ID:', error.message);
    throw error; // Rethrow the error for further handling
  }
};

const getEmrData = async (cycleId) => {
  try {
    console.log('Running query for EMR data with cycleId:', cycleId);
    const reports = await sequelize.query(
      `SELECT assignedDoctor, cycle_id FROM patient_counseling WHERE cycle_id = :cycleId`,
      { replacements: { cycleId }, type: sequelize.QueryTypes.SELECT }
    );
    console.log('Reports fetched:', reports);
    return reports;
  } catch (error) {
    console.error('Error fetching EMR data:', error.message);
    throw error;
  }
};
const getBillsByDateRange = async (mrNo, fromDate, toDate) => {
  try {
    console.log(`Fetching bills for MR No: ${mrNo} from ${fromDate} to ${toDate}`);
    const from = new Date(fromDate);
    const to = new Date(toDate);

    const transactionReceipts = await getTransactionReceipts(mrNo);
    const provisionalBills = await getProvisionalBills(mrNo);
    const serviceBills = await getServiceBills(mrNo);
    const pharmacyBills = await getPharmacyBills(mrNo);
    const packageBills = await getPackageBills(mrNo);
    const allBills = [
      ...provisionalBills,
      ...serviceBills,
      ...pharmacyBills,
      ...packageBills
    ];
    const allReceipts = transactionReceipts.filter(receipt => {
      const receiptDate = new Date(receipt.updatedAt);
      return receiptDate >= from && receiptDate <= to;
    });
    console.log("All Bills Before Filter:", allBills);
    const filteredBills = allBills.filter(bill => {
      const billDate = new Date(bill.date || bill.updatedAt);
      console.log("Bill Date:", to, billDate, from);

      return billDate >= from && billDate <= to;
    });
    console.log("Filtered Bills:", filteredBills);
    // response.send(filteredBills);
    return { filteredBills, allReceipts };
  } catch (error) {
    console.error('Error fetching bills by date range:', error.message);
    throw error;
  }
};

// BCExpMaster.sync({ alter: true });

const ConcessionMaster = sequelize.define(

  "ConcessionMaster",
  {
    userId: {
      type: DataTypes.INTEGER,
    },
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    exp_master_code: {
      type: DataTypes.STRING,
    },
    exp_master_desc: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "bcconcessionmaster", // Specify the table name
  }
);

// ConcessionMaster.sync({ alter: true });

const SacMaster = sequelize.define(
  "SacMaster",
  {
    userId: {
      type: DataTypes.INTEGER,
    },
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    exp_master_code: {
      type: DataTypes.STRING,
    },
    exp_master_desc: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "bcsacmaster", // Specify the table name
  }
);

// SacMaster.sync({ alter: true });
const RefundMaster = sequelize.define(
  "RefundMaster",
  {
    userId: {
      type: DataTypes.INTEGER,
    },
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    exp_master_code: {
      type: DataTypes.STRING,
    },
    exp_master_desc: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "bcrefundmaster", // Specify the table name
  }
);
// RefundMaster.sync({ alter: true });

const bcCompDetails = sequelize.define(
  "bcCompDetails",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    comp_code: {
      type: DataTypes.STRING,
    },
    comp_cont_person: {
      type: DataTypes.STRING,
    },
    comp_pers_role: {
      type: DataTypes.STRING,
    },
    comp_cont: {
      type: DataTypes.STRING,
    },
    comp_type: {
      type: DataTypes.STRING,
    },
    comp_email: {
      type: DataTypes.STRING,
    },
    comp_add: {
      type: DataTypes.STRING,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "companies", // Specify the table name
  }
);

const CompanyHF = sequelize.define(
  "CompanyHF",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    compHeaderContent: {
      type: DataTypes.TEXT,
    },
    compFooterContent: {
      type: DataTypes.TEXT,
    },
    comp_logo: {
      type: DataTypes.STRING,
    },
    head_image: {
      type: DataTypes.STRING,
    },
    comp_footer_image: {
      type: DataTypes.STRING,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "company_hf", // Specify the table name
  }
);

const ComTypeNew = sequelize.define(
  "comTypeNew",
  {
    code: {
      type: DataTypes.STRING,
      defaultValue: 0,
    },
    contact_person: {
      type: DataTypes.STRING,
    },
    description: {
      type: DataTypes.STRING,
    },
    contact_number: {
      type: DataTypes.STRING,
    },
    company_type: {
      type: DataTypes.STRING,
    },
    email_address: {
      type: DataTypes.STRING,
    },
    company_address: {
      type: DataTypes.STRING,
    },
    patient_source_category: {
      type: DataTypes.STRING,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "comtypenew", // Specify the table name
  }
);

// const bcService_Master = sequelize.define('bcService_Master', {
//   service_Code: {
//     type: DataTypes.STRING
//   },
//   service_Name: {
//     type: DataTypes.STRING
//   },
//   specialization: {
//     type: DataTypes.STRING
//   },
//   sub_Specialization: {
//     type: DataTypes.STRING
//   }
// }, {
//   timestamps: true, // Adds createdAt and updatedAt timestamps
//   alter: true,
//   tableName: 'bcService_Master' // Specify the table name
// });

const State = sequelize.define(
  "State",
  {
    stateCode: {
      type: DataTypes.STRING,
      primaryKey: true,
    },
    stateName: {
      type: DataTypes.STRING,
      allowNull: false,
    },
  },
  {
    timestamps: false,
    tableName: "state",
  }
);

const City = sequelize.define(
  "City",
  {
    cityName: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    stateCode: {
      type: DataTypes.STRING,
      references: {
        model: State,
        key: "stateCode",
      },
    },
  },
  {
    timestamps: false,
    tableName: "city",
  }
);

City.belongsTo(State, { foreignKey: "stateCode" });
State.hasMany(City, { foreignKey: "stateCode" });

const bcSerMasAssignConcent = sequelize.define(
  "bcSerMasAssignConcent",
  {
    department: {
      type: DataTypes.STRING,
    },
    template_name: {
      type: DataTypes.STRING,
      allowNull: false,
    },
  },
  {
    timestamps: false,
    tableName: "service-master-assign",
  }
);

const bcSerMastApplyLev = sequelize.define(
  "bcSerMastApplyLev",
  {
    l1: {
      type: DataTypes.STRING,
    },
    l2: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    l3: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    l4: {
      type: DataTypes.STRING,
      allowNull: false,
    },
  },
  {
    timestamps: false,
    tableName: "service-master-applay-level",
  }
);

const selectedDoctor = sequelize.define(
  "selectedDoctor",
  {
    Userid: {
      type: DataTypes.STRING,
    },
    name: {
      type: DataTypes.STRING,
    },
    spec: {
      type: DataTypes.STRING,
    },
    subSpec: {
      type: DataTypes.STRING,
    },
    subSpecs: {
      type: DataTypes.JSON, // Use JSONB or JSON depending on your needs and DB support
    },
    tariff: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: false,
    tableName: "selecteddoctor",
  }
);

const DoctorService = sequelize.define(
  "DoctorService",
  {
    UserId: {
      type: DataTypes.STRING,
    },
    doctorName: {
      type: DataTypes.STRING,
    },
    services: {
      type: DataTypes.JSON,
    },
  },
  {
    timestamps: true,
    tableName: "bcdoctorservice",
  }
);

const bcTariffMasterNew = sequelize.define(
  "bcTariffMasterNew",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
    },
    trf_code: {
      type: DataTypes.STRING,
    },
    trf_name: {
      type: DataTypes.STRING,
    },
    services: {
      type: DataTypes.JSON,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true,
    alter: true,
    tableName: "bctariffmasternew",
  }
);
// bcTariffMasterNew.sync()
const bcAdvAgent = sequelize.define(
  "bcAdvAgent",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
    },
    userId: {
      type: DataTypes.STRING,
    },
    adv_agnt_code: {
      type: DataTypes.STRING,
    },
    description: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true,
    tableName: "bcadvagent",
  }
);
// bcAdvAgent.sync({ alter: true });

// const BcTrfSer = sequelize.define('BcTrfSer', {
//   service_code: {
//     type: DataTypes.STRING,
//   },
//   service_name: {
//     type: DataTypes.STRING,
//     allowNull: false
//   },
//   specialization: {
//     type: DataTypes.STRING,
//     allowNull: false
//   },
//   sub_specialization: {
//     type: DataTypes.STRING,
//     allowNull: false
//   }
// }, {
//   timestamps: false,
//   tableName: 'BCTriff-Services'
// });

const BcAssCompNew = sequelize.define(
  "BcAssCompNew",
  {
    code: {
      type: DataTypes.STRING,
    },
    description: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    company: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    tariff: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true,
    tableName: "bcasscompnews",
  }
);

// BcAssCompNew.sync({ alter: true });

const Adnin = sequelize.define(
  "Admin",
  {
    userId: {
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true,
    },
    userName: {
      type: DataTypes.STRING,
    },
    userEmail: {
      type: DataTypes.STRING,
    },
    userMobile: {
      type: DataTypes.STRING,
    },
    userPassword: {
      type: DataTypes.STRING,
    },
    userRole: {
      type: DataTypes.STRING,
    },
    userStatus: {
      type: DataTypes.BOOLEAN,
    },
    clinicId: {
      type: DataTypes.STRING,
    },
  },
  {
    timestamps: true,
    alter: true,
    tableName: "admin",
  }
);

const bcBullRateChange = sequelize.define(
  "bcBullRateChange",
  {
    UserId: {
      type: DataTypes.STRING,
    },
    TariffCode: {
      type: DataTypes.STRING,
    },
    TariffName: {
      type: DataTypes.STRING,
    },
    remarks: {
      type: DataTypes.STRING,
    },
    effectiveDate: {
      type: DataTypes.DATE,
    },
    freeze: {
      type: DataTypes.BOOLEAN,
    },
    bulkRateChange: {
      type: DataTypes.JSON, // Assuming bulkRateChange is a JSON field
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true, // Adds createdAt and updatedAt timestamps
    alter: true,
    tableName: "bcbullratechange", // Specify the table name
  }
);

const Doctor = sequelize.define(
  "Doctor",
  {
    clinic_id: {
      type: DataTypes.STRING,
      defaultValue: "0",
    },
    doc_photo: {
      type: DataTypes.STRING,
    },
    doc_name: {
      type: DataTypes.STRING,
    },
    doc_dob: {
      type: DataTypes.DATE,
    },
    doc_spec: {
      type: DataTypes.STRING,
    },
    doc_sub_spec: {
      type: DataTypes.STRING,
    },
    doc_type: {
      type: DataTypes.STRING,
    },
    doc_catg: {
      type: DataTypes.STRING,
    },
    doc_mark_exec: {
      type: DataTypes.STRING,
    },
    doc_gender: {
      type: DataTypes.STRING,
    },
    doc_sign: {
      type: DataTypes.STRING,
    },
    doc_marital_status: {
      type: DataTypes.STRING,
    },
    doc_emp_no: {
      type: DataTypes.STRING,
    },
    doc_pf_no: {
      type: DataTypes.STRING,
    },
    doc_pan_no: {
      type: DataTypes.STRING,
    },
    doc_doj: {
      type: DataTypes.DATE,
    },
    doc_access_card_no: {
      type: DataTypes.STRING,
    },
    doc_reg_no: {
      type: DataTypes.STRING,
    },
    doc_email: {
      type: DataTypes.STRING,
    },
    doc_experience: {
      type: DataTypes.STRING,
    },
    doc_education: {
      type: DataTypes.STRING,
    },
  },
  {
    timestamps: true,
    tableName: "doctor",
  }
);

const SelectedService = sequelize.define(
  "SelectedService",
  {
    userId: {
      type: DataTypes.STRING,
    },
    docCategory: {
      type: DataTypes.STRING,
    },
    serviceName: {
      type: DataTypes.STRING,
    },
    className: {
      type: DataTypes.STRING,
    },
    serviceRate: {
      type: DataTypes.STRING,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true,
    tableName: "selectedservices",
  }
);

// SelectedService.sync({ alter: true });

const ServiceMasterSchema = sequelize.define(
  "tbl_service_master",
  {
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0
    },
    serviceCode: {
      type: DataTypes.STRING,
      allowNull: false,
      unique: true,
    },
    codeType: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    codeDetails: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    specialization: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    subSpecialization: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    sacCode: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    shortDescription: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    longDescription: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    serviceName: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    baseServiceRate: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    inHouse: {
      type: DataTypes.STRING,
    },
    inHouse_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    inHouse_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    service_tax: {
      type: DataTypes.STRING,
    },
    service_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    service_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    staffDiscount: {
      type: DataTypes.STRING,
    },
    staffDiscount_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    staffDiscount_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    staffParentAccount: {
      type: DataTypes.STRING,
    },
    staffParent_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    staffParent_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    concession: {
      type: DataTypes.STRING,
    },
    concession_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    concession_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    doctor: {
      type: DataTypes.STRING,
    },
    doctor_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    doctor_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    rateEditable: {
      type: DataTypes.STRING,
    },
    rateEditable_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    rateEditable_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    S_Citizen_Con: {
      type: DataTypes.STRING,
    },
    S_Citizen_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    S_Citizen_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    luxuryTax: {
      type: DataTypes.STRING,
    },
    luxuryTax_per: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    luxuryTax_amount: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    className: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    className_rate: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: false,
    alter: true,
    tableName: "services",
  }
);
// ServiceMasterSchema.sync();
const CompanyType = sequelize.define(
  "companyType",
  {
    userId: {
      type: DataTypes.INTEGER,
    },
    clinic_id: {
      type: DataTypes.INTEGER,
      defaultValue: 0,
    },
    code: {
      type: DataTypes.STRING,
    },
    description: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    status: {
      type: DataTypes.BOOLEAN,
      defaultValue: true,
    },
  },
  {
    timestamps: true,
    tableName: "companytype",
  }
);
// sequelize.sync();
module.exports = {
  CompanyHF,
  getServiceBills,
  getPharmacyBills,
  getCombinedBills,
  getTransactionReceipts,
  getReceiptDetails,
  getPatientIdFromMRNo,
  getSapData,
  getCycleIdFromPatientId,
  getEmrData,
  getBillsByDateRange,
  bcCompDetails,
  BCExpMaster,
  ServiceMaster,
  ComTypeNew,
  // bcService_Master,
  State,
  City,
  bcSerMasAssignConcent,
  bcSerMastApplyLev,
  bcTariffMasterNew,
  bcAdvAgent,
  // BcTrfSer,
  BcAssCompNew,
  Adnin,
  bcBullRateChange,
  Doctor,
  ServiceMasterSchema,
  SelectedService,
  CompanyType,
  selectedDoctor,
  ConcessionMaster,
  SacMaster,
  RefundMaster,
  DoctorService,
};
