
const express = require("express");
const { Op } = require("sequelize");
const { CrmFu_record, CrmApt_record, CrmVis_record, KYC, CrmPatientStatus_record } = require("../models/Kyc");



//? Get The Patient Status :--- 
const getStatus = async (req, res) => {
  const { status, date } = req.query;
  const LoginUser = req.user.username; // Logged-in user

  // console.log("LoginUser in getStatus", LoginUser);
  

  try {
    // Map the "cancel" status to "Cancelled" for the database query
    let whereCondition = {
      current_status: status.toLowerCase() === 'cancel' ? 'Cancelled' : status.toLowerCase(),
    };

    // Only apply date filter if status is NOT 'cancel'
    if (status.toLowerCase() !== "cancel" && date) {
      whereCondition.date = date;
    }

    let records = await CrmPatientStatus_record.findAll({
      attributes: ["id", "kyc_id", "clinic_id", "date", "lead_tag", "remark", "current_status", "updatedAt"],
      where: whereCondition,
      include: [
        {
          model: KYC,
          as: "kyc",
          attributes: ["id", "patient_name", "lead_owner"],
          where: { lead_owner: LoginUser }, // Filter by lead_owner
        },
      ],
      order: [["updatedAt", "DESC"]],
    });

    res.json({ status, records });
  } catch (error) {
    console.error("Error fetching records:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};


//? Get The Patient Journey :--- 
const getJourney = async (req, res) => {

  const { kyc_id } = req.query; // Get dynamic kyc_id from request
  const LoginUser = req.user.username;
  // console.log("LoginUser in getJourney", LoginUser);


  if (!kyc_id) {
    return res.status(400).json({ error: "kyc_id is required" });
  }

  try {
    let records = [];

    // Fetch Follow-up & Appointment from crmfu_record
    const crmFuRecords = await CrmFu_record.findAll({
      attributes: ["id", "kyc_id", "remark", "fu_next_call", "lead_tag", "fu_status", "updatedAt"],
      where: { kyc_id: kyc_id }, // Filter by kyc_id
      include: [
        {
          model: KYC,
          as: "kyc",
          attributes: ["id", "patient_name", "lead_owner"],
          where: { lead_owner: LoginUser },
        },
      ],
      order: [["updatedAt", "DESC"]],
    });

    const formattedFuRecords = crmFuRecords.map((rec) => ({
      patient_name: rec.kyc.patient_name,
      status: rec.fu_status.toLowerCase() === "follow up" ? "Follow-up" : "Appointment",
      remark: rec.remark || "N/A",
      lead_tag: rec.lead_tag,
      lead_owner: rec.kyc.lead_owner,
      updatedAt: rec.updatedAt,
    }));

    // Fetch Visited, Cancelled, Rescheduled, and Pending Appointments from crmapt_record
    const crmAptRecords = await CrmApt_record.findAll({
      attributes: ["id", "kyc_id", "clinic", "doctor", "apt_date", "lead_tag", "apt_status", "updatedAt"],
      where: { kyc_id: kyc_id }, // Filter by kyc_id
      include: [
        {
          model: KYC,
          as: "kyc",
          attributes: ["id", "patient_name", "lead_owner"],
          where: { lead_owner: LoginUser },
        },
      ],
      order: [["updatedAt", "DESC"]],
    });

    const formattedAptRecords = crmAptRecords.map((rec) => ({
      patient_name: rec.kyc.patient_name,
      status: rec.apt_status ? rec.apt_status : "Pending",
      remark: rec.remark || "N/A",
      lead_tag: rec.lead_tag,
      lead_owner: rec.kyc.lead_owner,
      updatedAt: rec.updatedAt,
    }));

    // Fetch Booked & Not Booked from crmvis_record
    const crmVisRecords = await CrmVis_record.findAll({
      attributes: ["id", "kyc_id", "remark", "visited_status", "lead_tag", "updatedAt"],
      where: { kyc_id: kyc_id }, // Filter by kyc_id
      include: [
        {
          model: KYC,
          as: "kyc",
          attributes: ["id", "patient_name", "lead_owner"],
          where: { lead_owner: LoginUser },
        },
      ],
      order: [["updatedAt", "DESC"]],
    });

    const formattedVisRecords = crmVisRecords.map((rec) => ({
      patient_name: rec.kyc.patient_name,
      status: rec.visited_status,
      lead_tag: rec.lead_tag,
      remark: rec.remark || "N/A",
      lead_owner: rec.kyc.lead_owner,
      updatedAt: rec.updatedAt,
    }));

    // Combine all records
    records = [...formattedFuRecords, ...formattedAptRecords, ...formattedVisRecords];

    // Sort records by updatedAt in descending order (latest first)
    records.sort((a, b) => new Date(a.updatedAt) - new Date(b.updatedAt));

    // console.log("Total records fetched:", records.length);
    res.json({ status: "success", records });
  } catch (error) {
    console.error("Error fetching patient journey:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};


//? Get Current Status 
const getCurrentStatus = async (req, res) => {
  const { kyc_id } = req.query; // Get dynamic kyc_id from request
  const LoginUser = req.user.username;

  // console.log("Login User in the getCurrentStatus", LoginUser);
  

  if (!kyc_id) {
    return res.status(400).json({ error: "kyc_id is required" });
  }

  try {
    let statusRecords = [];

    // Fetch current patient status from CrmPatientStatus_record
    const crmPatientStatusRecords = await CrmPatientStatus_record.findAll({
      attributes: ["id", "kyc_id", "lead_tag", "current_status", "updatedAt"],
      where: { kyc_id: kyc_id }, // Filter by kyc_id
      include: [
        {
          model: KYC,
          as: "kyc",
          attributes: ["id", "patient_name", "lead_owner"],
          where: { lead_owner: LoginUser },
        },
      ],
      order: [["updatedAt", "DESC"]],
    });

    // Format records
    statusRecords = crmPatientStatusRecords.map((rec) => ({
      patient_name: rec.kyc.patient_name,
      status: rec.current_status,
      lead_tag: rec.lead_tag,
      lead_owner: rec.kyc.lead_owner,
      updatedAt: rec.updatedAt,
    }));

    // console.log("Total status records fetched:", statusRecords.length);
    res.json({ status: "success", statusRecords });
  } catch (error) {
    console.error("Error fetching patient status records:", error.message, error.stack);
    res.status(500).json({ error: "Internal server error", details: error.message });
  }
};



module.exports = { getStatus, getJourney, getCurrentStatus };
