const axios = require('axios');
const { Leads } = require('../models/Kyc'); // Import your Leads model

// Constants for Facebook API
const PAGE_ID = process.env.PAGE_ID || "360040977191465";
const PAGE_ACCESS_TOKEN = process.env.PAGE_ACCESS_TOKEN || "YOUR_FACEBOOK_ACCESS_TOKEN";

// Field mappings for Facebook leads
const emailList = ["Email ID", "work_email", "email", "email_id", "Email_ID", "EMAIL_ID"];
const phoneList = ["Phone Number", "work_phone_number", "phone_number", "Phone_Number"];
const nameList = ["Full name", "full_name", "name", "FULL_NAME", "full name", "Full_name", "Full_Name"];

/**
 * Fetch and store Facebook leads
 */
const fetchAndStoreFacebookLeads = async () => {
  try {
    const response = await axios.get(`https://graph.facebook.com/v16.0/${PAGE_ID}/leadgen_forms`, {
      params: { access_token: PAGE_ACCESS_TOKEN, fields: "id,name" },
    });

    const leadFormsData = response.data.data;

    for (const form of leadFormsData) {
      const formId = form.id;
      const leadsResponse = await axios.get(`https://graph.facebook.com/v16.0/${formId}/leads`, {
        params: { access_token: PAGE_ACCESS_TOKEN },
      });

      const leadsData = leadsResponse.data.data;

      for (const lead of leadsData) {
        const email = lead.field_data.find(f => emailList.includes(f.name))?.values[0] || null;
        const phone = lead.field_data.find(f => phoneList.includes(f.name))?.values[0] || null;
        const fullName = lead.field_data.find(f => nameList.includes(f.name))?.values[0] || null;

        // Store data in the Leads table
        await Leads.create({
          patient_name: fullName,
          patient_email: email,
          phone_1: phone,
          source: 'Facebook',
          lead_date: new Date(),
        });
      }
    }
    console.log("Facebook leads fetched and stored successfully.");
  } catch (error) {
    console.error("Error fetching or storing Facebook leads:", error);
  }
};

/**
 * Submit a lead (generic function for all sources)
 */
const submitLead = async (req, res) => {
  console.log("You are in the SubmitLead");

  try {
    const {
      patient_name,
      patient_email,
      phone_1,
      utm_source,
      utm_medium,
      utm_term,
      utm_content,
      utm_campaign,
      fbclid,
      gclid,
      ...extra
    } = req.body;

    // Validate required fields
    if (!patient_name || !patient_email) {
      return res.status(400).json({ message: "patient_name and patient_email are required fields" });
    }

    // Determine the source dynamically
    let source = "other";
    if (fbclid) source = "facebook";
    else if (gclid) source = "google";
    else if (utm_source) source = utm_source.toLowerCase();

    // Store data in the Leads table
    await Leads.create({
      patient_name,
      patient_email,
      phone_1: phone_1 || null, // Optional field
      source,
      lead_date: new Date(), // Use current date
      utm_source,
      utm_medium,
      utm_term,
      utm_content,
      utm_campaign,
      extra_data: extra, // Store additional fields
    });

    res.status(201).json({ message: "Lead stored successfully" });
  } catch (error) {
    console.error("Error storing lead:", error);
    res.status(500).json({ message: "Error storing lead" });
  }
};

/**
 * Start periodic Facebook lead fetching
 */
const startPeriodicLeadFetch = () => {
  setInterval(async () => {
    try {
      console.log('Fetching and storing Facebook leads...');
      await fetchAndStoreFacebookLeads();
      console.log('Facebook leads fetched and stored successfully.');
    } catch (error) {
      console.error('Error in periodic Facebook lead fetch:', error);
    }
  }, 1000 * 60 * 30); // Run every 30 minutes
};

// Start the periodic lead fetch when the controller is initialized
startPeriodicLeadFetch();

module.exports = {
  fetchAndStoreFacebookLeads,
  submitLead,
};