const { ClinicConfiguration } = require("../models/clinicConfig");
const { PatientCounseling, Package, Bifurcation_Stages, Outcome } = require("../models/embrology");
const { NewPackage } = require("../models/packageConfig");
const { PR_patientReg, PR_BillFindPatient, PR_BillPharmacy } = require("../models/PatientReg");
const { kycCtrl } = require("./controller");
const { KYC, Leads, CrmPatientStatus_record, CrmVis_record } = require('../models/Kyc');
const { PatientRefund, CompanyRefund } = require('../models/MainBill')


const getClinicsForUser = async (userClinicId, role) => {
  if (role === 1) {
    // If the user is an admin, fetch all clinics under the main clinic
    const subClinics = await ClinicConfiguration.findAll({
      where: { clinic_id: userClinicId },
    });

    // Return the clinic ID for the main clinic plus all sub-clinic IDs
    return [userClinicId, ...subClinics.map(c => c.id)];
  }

  // If the user is a doctor, they only have access to their own clinic
  return [userClinicId];
};

const getPatientDetailsCounseling = async (req, res) => {
  try {
    const clinicIds = await getClinicsForUser(req.user.clinicId, req.user.role);
    // Step 1: Fetch all PatientCounseling data, including patientId, packageId, and amount
    const patientCounselingData = await PatientCounseling.findAll({
      where: { clinic_id: clinicIds },
      attributes: ['patientId', 'package_id', 'amount'], // Only fetching relevant fields
    });

    // Step 2: For each PatientCounseling entry, retrieve patient name and package details
    const result = [];

    for (const counseling of patientCounselingData) {
      const patientId = counseling.patientId;
      const packageId = counseling.package_id;

      // Fetch Patient name by patientId
      const patient = await PR_patientReg.findOne({
        where: { id: patientId },
        attributes: ['firstName', 'lastName', 'mr_no'], // Assuming these are the fields in the Patient model
      });

      // If no patient is found, skip this iteration
      if (!patient) {
        continue;
      }

      // Fetch Package details by packageId
      const package = await NewPackage.findOne({
        where: { id: packageId },
        attributes: ['packageName', 'packageRate'], // Assuming these are the fields in the Package model
      });

      // If no package is found, skip this iteration
      if (!package) {
        continue;
      }

      // Step 3: Format the response with required details
      result.push({
        patientName: `${patient.firstName} ${patient.lastName}`,
        mr_no: patient.mr_no, // Combining first and last name
        packageName: package.packageName,
        packageRate: package.packageRate,
        amount: counseling.amount, // From PatientCounseling
      });
    }

    // Step 4: Send the final response
    res.status(200).json(result);
  } catch (error) {
    console.error('Error fetching patient details counseling:', error);
    res.status(500).json({ msg: 'An error occurred while fetching patient details counseling.' });
  }
};


const getClinicalKpiReports = async (req, res) => {
  try {
    const patients = await PR_patientReg.findAll({
      attributes: ["id", "mr_no", "lead_no","firstName","email", "phone1", "phone2", "createdAt", "city", "doctor", "spouse_firstName"],
      include: [
        {
          model: KYC,
          as: "kycDetails",
          attributes: [
            "city",
            "contact_no_husband",
            "contact_no_wife",
            "email_id",
            "lead_no",
            "lead_owner",
            "patient_name",
            "source",
            "spouce_name",
            "createdAt"
          ],
          include: [
            {
              model: Leads,
              as: "lead",
              attributes: { exclude: ["createdAt", "updatedAt"] },
            },
            {
              model: CrmPatientStatus_record,
              as: "crmPatientStatusRecords",
              attributes: ["current_status"],
            },
            {
              model: CrmVis_record,
              as: "crmVisRecords",
              attributes: ["clinic", "doctor", "package", "treatment", "uhid_no", "visited_status"]
            }
          ],
        },
        {
          model: PR_BillFindPatient,
          as: "billFindPatients",
          attributes: ["bill_no", "clinicBillAmount", "mrNo", "totalBillAmount", "selectedBillServices", "totalNetAmount"],

        },
        {
          model: PR_BillPharmacy,
          as: "billPharmacies",
          attributes: ["totalNetAmount", "balanceAmount", "Advance", "bill_no", "mrNo"],
        },
        {
          model: PatientRefund,
          as: "refunds",
          attributes: ["refundAmount", "rec_no", "patient_id"],
        },
        {
          model: Bifurcation_Stages,
          as: "bifurcationStages",
        },
        {
          model: PatientCounseling,
          as: "patientCounseling",
          attributes: ["currentPage", "status", "cycle_id", "patientId", "createdAt"],
          include: [
            {
              model: Outcome,
              as: "Outcome",
              attributes: [
                "clinicalPregnancyDate",
                "clinicalPregnancyRecordedBy",
                "biochemicalPregnancy",
                "miscarriageDate",
                "modeOfDelivery",
                "birthType",
              ],
            },
          ],
        },
        {
          model: CompanyRefund,
          as: "companyRefunds",
          attributes: ["patient_id", "refundAmount", "totalAdvance",],
        }
      ],
    });


    console.log("Your Data is Here ", patients);


    return res.status(200).json({
      success: true,
      data: patients,
    });
  } catch (error) {
    console.error("Error fetching clinical KPI reports:", error);
    return res.status(500).json({
      success: false,
      message: "Failed to fetch clinical KPI reports.",
      error: error.message,
    });
  }
};


module.exports = { getPatientDetailsCounseling, getClinicsForUser, getClinicalKpiReports }  