// routes/noteRoutes.js
const express = require("express");
const router = express.Router();
const { my_Notes: Note } = require("../models/notes");

// Get all notes for a specific clinic
router.get("/notes", async (req, res) => {
  try {
    console.log("Session give: ", req.user);
    const notes = await Note.findAll({
      where: { ClinicId: req.user.clinicId },
      order: [["createdAt", "DESC"]],
    });
    res.json(notes);
  } catch (error) {
    console.error("Error fetching notes:", error);
    res.status(500).json({ error: error.message });
  }
});

// Get a single note
router.get("/notes/:id", async (req, res) => {
  try {
    const note = await Note.findByPk(req.params.id);
    if (!note) {
      return res.status(404).json({ error: "Note not found" });
    }
    res.json(note);
  } catch (error) {
    console.error("Error fetching note:", error);
    res.status(500).json({ error: error.message });
  }
});

// Create a new note
router.post("/notes", async (req, res) => {
  try {
    console.log("logging user: ", req.user);

    const { title, description, x_coordinate, y_coordinate, color } = req.body;

    const note = await Note.create({
      title,
      description,
      x_coordinate,
      y_coordinate,
      color,
      ClinicId: req.user.clinicId,
    });

    res.status(201).json(note);
  } catch (error) {
    console.error("Error creating note:", error);
    res.status(500).json({ error: error.message });
  }
});

// Update a note
router.put("/notes/:id", async (req, res) => {
  try {
    const { title, description, x_coordinate, y_coordinate, color } = req.body;

    const note = await Note.findByPk(req.params.id);
    if (!note) {
      return res.status(404).json({ error: "Note not found" });
    }

    await note.update({
      title,
      description,
      x_coordinate,
      y_coordinate,
      color,
    });

    res.json(note);
  } catch (error) {
    console.error("Error updating note:", error);
    res.status(500).json({ error: error.message });
  }
});

// Delete a note
router.delete("/notes/:id", async (req, res) => {
  try {
    const note = await Note.findByPk(req.params.id);
    if (!note) {
      return res.status(404).json({ error: "Note not found" });
    }

    await note.destroy();
    res.json({ message: "Note deleted successfully" });
  } catch (error) {
    console.error("Error deleting note:", error);
    res.status(500).json({ error: error.message });
  }
});

// Bulk update notes positions (for when multiple notes are moved)
router.put("/notes/bulk-update", async (req, res) => {
  try {
    const { notes } = req.body;

    // Use transaction to ensure all updates succeed or none do
    await sequelize.transaction(async (t) => {
      for (const noteData of notes) {
        await Note.update(
          {
            x_coordinate: noteData.x_coordinate,
            y_coordinate: noteData.y_coordinate,
          },
          {
            where: { id: noteData.id },
            transaction: t,
          }
        );
      }
    });

    res.json({ message: "Notes updated successfully" });
  } catch (error) {
    console.error("Error bulk updating notes:", error);
    res.status(500).json({ error: error.message });
  }
});

module.exports = router;
